<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_liste
 *
 * @copyright   Copyright (C) 2025 Votre Nom. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;

// Récupérer la base URL du site
$root = Uri::root();
$app = Factory::getApplication();

// Paramètres de pagination
$itemsPerPage = (int)$params->get('items_per_page', 5); // Nombre d'agriculteurs par page
$currentPage = max(1, $app->input->getInt('page', 1)); // Page actuelle
$startIndex = ($currentPage - 1) * $itemsPerPage;
$totalPages = ceil(count($agriculteurs) / $itemsPerPage);

// Détecter la vue active (liste ou grille) à partir des paramètres d'URL
$activeView = $app->input->getCmd('view_mode', 'list'); // Par défaut: liste

// Récupérer la catégorie à filtrer depuis l'URL
$filterCategory = $app->input->getString('category', '');

// Sous-ensemble d'agriculteurs pour la page actuelle
$pageAgriculteurs = array_slice($agriculteurs, $startIndex, $itemsPerPage);

// Construire l'URL pour la pagination avec SP Page Builder et préserver la vue active et la catégorie
$queryParams = [
    'option' => 'com_sppagebuilder', 
    'view' => 'page', 
    'id' => $params->get('sppb_page_id', '50'), 
    'view_mode' => $activeView
];

// Ajouter le paramètre de catégorie s'il existe
if (!empty($filterCategory)) {
    $queryParams['category'] = $filterCategory;
}

$baseUrl = $root . 'index.php?' . http_build_query(array_merge(
    $queryParams,
    array_diff_key($_GET, array_flip(['option', 'view', 'id', 'view_mode', 'category', 'page']))
));

// URL spécifique à votre site pour les fiches agriculteurs
$profileUrlBase = $root . 'index.php/nos-agriculteurs';
?>

<div class="agriculteur-liste-container">
    <!-- Popup de recherche -->
    <div id="search-popup" class="search-popup">
        <div class="search-popup-content">
            <h2>Rechercher</h2>
            <div class="search-form">
                <input type="text" id="search-input" placeholder="Entrez un ou plusieurs mots..." autocomplete="off">
                <button id="search-submit">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <button id="search-close" class="search-close-button">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </div>

    <!-- Barre de navigation supérieure avec boutons -->
    <div class="top-navigation-bar">
        <a href="index.php?option=com_sppagebuilder&view=page&id=52" class="nav-button back-button">
            <i class="fas fa-arrow-left"></i>
        </a>
        <div class="view-buttons">
            <button id="map-view-button" class="nav-button">
                <i class="fas fa-map-marked-alt"></i>
            </button>
            <button id="grid-view-button" class="nav-button <?php echo ($activeView == 'grid') ? 'active' : ''; ?>">
                <i class="fas fa-th"></i>
            </button>
            <button id="search-button" class="nav-button">
                <i class="fas fa-search"></i>
            </button>
        </div>
    </div>
    
    <!-- Conteneur pour la vue carte (initialement caché) -->
    <div id="map-container" class="map-container" style="display: none;">
        <div id="google-map" class="google-map"></div>
    </div>
    
    <!-- Conteneur pour la vue grille (affichée par défaut) -->
    <div id="grid-container" class="grid-container">
        <!-- Bouton de géolocalisation -->
        <button id="geolocate-button-<?php echo $module->id; ?>" class="geolocation-button">
            <i class="fas fa-location-crosshairs"></i> Les plus proches
        </button>
        
        <div id="distances-info-<?php echo $module->id; ?>" class="distances-info" style="display: none;">
            <i class="fas fa-info-circle"></i> 
            Distances calculées depuis votre position actuelle.
        </div>
        
        <!-- Vue liste -->
        <div id="list-view" class="view-container <?php echo ($activeView == 'list') ? 'active-view' : ''; ?>">
            <ul class="farmer-list" id="farmer-list-<?php echo $module->id; ?>">
                <?php foreach ($pageAgriculteurs as $agriculteur): ?>
                <li>
                    <?php 
                    // Construction du lien vers le profil avec l'URL exacte de votre site
                    // Ajout des paramètres page_retour, view_mode et category pour conserver le contexte
                    $profileUrl = $profileUrlBase . '?submission_id=' . $agriculteur['id'] 
                        . '&page_retour=' . $currentPage 
                        . '&view_mode=' . $activeView;
                    
                    // Ajouter le paramètre de catégorie s'il existe
                    if (!empty($filterCategory)) {
                        $profileUrl .= '&category_retour=' . urlencode($filterCategory);
                    }
                    
                    // Vérifier si l'image existe
                    $imageUrl = !empty($agriculteur['image']) ? $agriculteur['image'] : $root . 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                    ?>
                    <a href="<?php echo $profileUrl; ?>" class="farmer-item" 
                       data-coordinates="<?php echo htmlspecialchars($agriculteur['coordinates']); ?>"
                       data-id="<?php echo $agriculteur['id']; ?>"
                       data-image="<?php echo $imageUrl; ?>"
                       data-name="<?php echo htmlspecialchars($agriculteur['societe']); ?>"
                       data-owner="<?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?>"
                       <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                       data-categories="<?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?>"
                       <?php endif; ?>
                       <?php if (isset($agriculteur['description'])): ?>
                       data-description="<?php echo htmlspecialchars($agriculteur['description']); ?>"
                       <?php endif; ?>
                       <?php if (isset($agriculteur['location'])): ?>
                       data-location="<?php echo htmlspecialchars($agriculteur['location']); ?>"
                       <?php endif; ?>
                      >
                        <div class="farmer-content">
                            <img src="<?php echo $imageUrl; ?>" alt="<?php echo htmlspecialchars($agriculteur['societe']); ?>" class="farmer-image" onerror="this.src='<?php echo $root; ?>media/mod_agriculteur_liste/images/default-farmer.jpg'">
                            <div class="farmer-info">
                                <h3 class="farmer-name"><?php echo htmlspecialchars($agriculteur['societe']); ?></h3>
                                <p class="farmer-owner"><?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?></p>
                                <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                                <p class="farmer-categories"><?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?></p>
                                <?php endif; ?>
                            </div>
                            <div class="farmer-distance" id="distance-<?php echo $agriculteur['id']; ?>">
                                <p class="distance-value"><?php echo number_format($agriculteur['distance'], 1, '.', ''); ?></p>
                                <p class="distance-unit">km</p>
                            </div>
                        </div>
                    </a>
                </li>
                <?php endforeach; ?>
            </ul>
        </div>
        
        <!-- Vue grille (masquée, sera affichée en JS) -->
        <div id="grid-view" class="view-container <?php echo ($activeView == 'grid') ? 'active-view' : ''; ?>">
            <div class="farmer-grid" id="farmer-grid-<?php echo $module->id; ?>">
                <?php foreach ($pageAgriculteurs as $agriculteur): ?>
                <?php 
                // Construction du lien vers le profil avec l'URL exacte de votre site
                // Ajout des paramètres page_retour, view_mode et category pour conserver le contexte
                $profileUrl = $profileUrlBase . '?submission_id=' . $agriculteur['id'] 
                    . '&page_retour=' . $currentPage 
                    . '&view_mode=' . $activeView;
                
                // Ajouter le paramètre de catégorie s'il existe
                if (!empty($filterCategory)) {
                    $profileUrl .= '&category_retour=' . urlencode($filterCategory);
                }
                
                // Vérifier si l'image existe
                $imageUrl = !empty($agriculteur['image']) ? $agriculteur['image'] : $root . 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                ?>
                <a href="<?php echo $profileUrl; ?>" class="grid-item" 
                   data-coordinates="<?php echo htmlspecialchars($agriculteur['coordinates']); ?>"
                   data-id="<?php echo $agriculteur['id']; ?>"
                   data-image="<?php echo $imageUrl; ?>"
                   data-name="<?php echo htmlspecialchars($agriculteur['societe']); ?>"
                   data-owner="<?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?>"
                   <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                   data-categories="<?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?>"
                   <?php endif; ?>
                   <?php if (isset($agriculteur['description'])): ?>
                   data-description="<?php echo htmlspecialchars($agriculteur['description']); ?>"
                   <?php endif; ?>
                   <?php if (isset($agriculteur['location'])): ?>
                   data-location="<?php echo htmlspecialchars($agriculteur['location']); ?>"
                   <?php endif; ?>
                  >
                    <div class="grid-item-content">
                        <div class="grid-item-image-container">
                            <img src="<?php echo $imageUrl; ?>" alt="<?php echo htmlspecialchars($agriculteur['societe']); ?>" class="grid-item-image" onerror="this.src='<?php echo $root; ?>media/mod_agriculteur_liste/images/default-farmer.jpg'">
                            <div class="grid-item-distance" id="grid-distance-<?php echo $agriculteur['id']; ?>">
                                <span class="grid-distance-value"><?php echo number_format($agriculteur['distance'], 1, '.', ''); ?></span>
                                <span class="grid-distance-unit">km</span>
                            </div>
                        </div>
                        <div class="grid-item-info">
                            <h3 class="grid-item-name"><?php echo htmlspecialchars($agriculteur['societe']); ?></h3>
                            <p class="grid-item-owner"><?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?></p>
                            <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                            <p class="grid-item-categories"><?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                </a>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- Conteneur des résultats de recherche -->
        <div id="search-results-container" class="view-container" style="display: none;">
            <h3 class="search-results-title">Résultats de recherche</h3>
            <p id="search-count" class="search-count"></p>
            <ul class="farmer-list" id="search-results-list">
                <!-- Les résultats de recherche seront injectés ici par JavaScript -->
            </ul>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="pagination-container">
            <div class="pagination">
                <!-- Retour à la première page -->
                <?php if ($currentPage > 1): ?>
                <a href="<?php echo $baseUrl . '&page=1'; ?>" class="page-item first">
                    <i class="fas fa-angle-double-left"></i>
                </a>
                <?php endif; ?>
                
                <!-- Page précédente -->
                <?php if ($currentPage > 1): ?>
                <a href="<?php echo $baseUrl . '&page=' . ($currentPage - 1); ?>" class="page-item prev">
                    <i class="fas fa-chevron-left"></i>
                </a>
                <?php endif; ?>
                
                <?php
                // Afficher les numéros de page avec limitation
                $maxPagesToShow = 5;
                $startPage = max(1, min($currentPage - floor($maxPagesToShow / 2), $totalPages - $maxPagesToShow + 1));
                $endPage = min($totalPages, $startPage + $maxPagesToShow - 1);
                
                // Ajuster startPage si nécessaire
                $startPage = max(1, $endPage - $maxPagesToShow + 1);
                
                // Afficher la première page si nécessaire
                if ($startPage > 1): ?>
                    <a href="<?php echo $baseUrl . '&page=1'; ?>" class="page-item">1</a>
                    <?php if ($startPage > 2): ?>
                    <span class="page-item ellipsis">...</span>
                    <?php endif; ?>
                <?php endif; ?>
                
                <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <?php if ($i == $currentPage): ?>
                    <span class="page-item active"><?php echo $i; ?></span>
                    <?php else: ?>
                    <a href="<?php echo $baseUrl . '&page=' . $i; ?>" class="page-item"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php 
                // Afficher la dernière page si nécessaire
                if ($endPage < $totalPages): ?>
                    <?php if ($endPage < $totalPages - 1): ?>
                    <span class="page-item ellipsis">...</span>
                    <?php endif; ?>
                    <a href="<?php echo $baseUrl . '&page=' . $totalPages; ?>" class="page-item"><?php echo $totalPages; ?></a>
                <?php endif; ?>
                
                <!-- Page suivante -->
                <?php if ($currentPage < $totalPages): ?>
                <a href="<?php echo $baseUrl . '&page=' . ($currentPage + 1); ?>" class="page-item next">
                    <i class="fas fa-chevron-right"></i>
                </a>
                <?php endif; ?>
            </div>
            
            <div class="page-info">
                Page <?php echo $currentPage; ?> sur <?php echo $totalPages; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<style>
/* Styles pour le fond de la page */
body, html {
    margin: 0;
    padding: 0;
    min-height: 100vh;
    background-image: url('/media/mod_agriculteur_liste/images/fond-vert.jpg');
    background-size: cover;
    background-position: center;
    background-attachment: fixed;
}

.agriculteur-liste-container {
    position: relative;
    min-height: 100vh;
}

/* Styles pour le popup de recherche */
.search-popup {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.7);
    display: none;
    justify-content: center;
    align-items: center;
    z-index: 2000;
    backdrop-filter: blur(5px);
}

.search-popup-content {
    position: relative;
    background-color: white;
    padding: 30px;
    border-radius: 10px;
    width: 90%;
    max-width: 500px;
    box-shadow: 0 4px 15px rgba(0, 0, 0, 0.2);
}

.search-popup-content h2 {
    text-align: center;
    color: #356d21;
    margin-top: 0;
    margin-bottom: 20px;
}

.search-form {
    display: flex;
    width: 100%;
}

#search-input {
    flex-grow: 1;
    padding: 12px 15px;
    border: 1px solid #ddd;
    border-radius: 5px 0 0 5px;
    font-size: 16px;
    outline: none;
}

#search-input:focus {
    border-color: #3c8f44;
}

#search-submit {
    background-color: #3c8f44;
    color: white;
    border: none;
    padding: 12px 15px;
    font-size: 16px;
    border-radius: 0 5px 5px 0;
    cursor: pointer;
    transition: background-color 0.2s;
}

#search-submit:hover {
    background-color: #356d21;
}

.search-close-button {
    position: absolute;
    top: 10px;
    right: 10px;
    background: none;
    border: none;
    color: #666;
    font-size: 18px;
    cursor: pointer;
}

/* Styles pour les résultats de recherche */
.search-results-title {
    margin: 10px 0;
    color: #3c8f44;
    font-size: 1.2rem;
}

.search-count {
    margin-bottom: 15px;
    font-size: 0.9rem;
    color: #666;
}

.no-results {
    padding: 20px;
    text-align: center;
    background-color: rgba(255, 255, 255, 0.85);
    border-radius: 5px;
    margin: 20px 0;
}

/* Styles pour la barre de navigation supérieure */
.top-navigation-bar {
    display: flex;
    justify-content: space-between;
    align-items: center;
    background-color: #356d21; /* Vert foncé comme sur le screenshot */
    padding: 10px 15px;
    position: sticky;
    top: 0;
    z-index: 1000;
    color: white;
    margin-bottom: 10px;
    border-radius: 10px;
}

.nav-button {
    background: none;
    border: none;
    color: white;
    font-size: 1.4rem;
    cursor: pointer;
    padding: 8px;
    display: inline-flex;
    justify-content: center;
    align-items: center;
    position: relative;
}

.nav-button.active {
    color: #ffcc00; /* Bouton actif en jaune */
}

.back-button {
    font-size: 1.6rem;
    text-decoration: none;
}

.view-buttons {
    display: flex;
    gap: 15px;
}

/* Styles pour l'icône logo */
.logo-reunion {
    position: absolute;
    left: 50%;
    transform: translateX(-50%);
    width: 40px;
    height: 40px;
}

/* Conteneur de la carte Google Maps */
.map-container {
    width: 100%;
    height: 75vh;
    margin-bottom: 20px;
    border-radius: 10px;
    overflow: hidden;
}

.google-map {
    width: 100%;
    height: 100%;
    background-color: #f0f0f0;
    border-radius: 10px;
}

/* Styles pour l'infowindow */
.custom-infowindow {
    max-width: 300px;
    overflow: hidden;
    border-radius: 5px;
}

.infowindow-image-container {
    position: relative;
    width: 100%;
    padding-top: 56.25%; /* Ratio 16:9 */
    overflow: hidden;
}

.infowindow-image {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.infowindow-content {
    padding: 10px;
}

.infowindow-title {
    font-size: 1.1rem;
    font-weight: bold;
    margin: 0 0 5px 0;
    color: #3c8f44;
}

.infowindow-subtitle {
    font-size: 0.9rem;
    margin: 0 0 10px 0;
    color: #666;
}

.infowindow-link {
    display: inline-block;
    padding: 5px 10px;
    background-color: #3c8f44;
    color: white !important;
    text-decoration: none !important;
    border-radius: 3px;
    font-size: 0.85rem;
    margin-top: 5px;
    transition: background-color 0.2s;
}

.infowindow-link:hover {
    background-color: #2a6f30;
    text-decoration: none !important;
    color: white !important;
}

/* Styles pour le cluster marker */
.cluster-marker {
    background-color: rgba(60, 143, 68, 0.7);
    border: 2px solid #3c8f44;
    border-radius: 50%;
    color: white;
    height: 40px;
    width: 40px;
    line-height: 37px;
    text-align: center;
    font-weight: bold;
    font-size: 14px;
}

/* Conteneur des vues (liste et grille) */
.view-container {
    display: none;
}

.active-view {
    display: block;
}

/* Styles pour la vue grille */
.farmer-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr);
    gap: 10px;
    margin: 0;
    padding: 0;
}

.grid-item {
    background-color: rgba(255, 255, 255, 0.85);
    border-radius: 5px;
    backdrop-filter: blur(3px);
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    text-decoration: none !important;
    color: inherit !important;
    position: relative;
    display: flex;
    flex-direction: column;
    transition: transform 0.2s;
}

.grid-item:active {
    transform: scale(0.92);
}

.grid-item-content {
    display: flex;
    flex-direction: column;
    height: 100%;
    padding: 5px;
}

.grid-item-image-container {
    position: relative;
    padding-top: 75%; /* Ratio 4:3 */
    overflow: hidden;
    position: relative;
    overflow: hidden;
    margin-bottom: 5px;
}

.grid-item-image-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(255, 255, 255, 0.1);
    border-radius: 50%;
    transform: scale(0);
    transition: transform 0.5s cubic-bezier(0.25, 0.46, 0.45, 0.94);
}

.grid-item:active .grid-item-image-container::before {
    transform: scale(1.5);
    opacity: 0;
    transition: transform 0.6s, opacity 0.6s;
}

.grid-item-image {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    object-fit: cover;
    object-position: center;
    transition: transform 0.3s;
}

.grid-item:active .grid-item-image {
    transform: scale(0.9);
}

.grid-item-distance {
    position: absolute;
    bottom: 10px;
    right: 10px;
    background-color: rgba(255, 255, 255, 0.85);
    padding: 3px 8px;
    border-radius: 12px;
    font-size: 0.8rem;
    display: flex;
    flex-direction: column;
    align-items: center;
    line-height: 1;
}

.grid-distance-value {
    font-weight: bold;
    color: #333;
    font-size: 0.9rem;
}

.grid-distance-unit {
    font-size: 0.7rem;
    color: #666;
}

.grid-item-info {
    padding: 10px;
    flex-grow: 1;
    display: flex;
    flex-direction: column;
    justify-content: space-between;
}

.grid-item-name {
    font-weight: bold !important;
    color: #34643d !important;
    margin: 0 0 5px 0 !important;
    font-size: 0.9rem !important;
    overflow: hidden;
    text-overflow: ellipsis;
    display: -webkit-box;
    -webkit-line-clamp: 2;
    -webkit-box-orient: vertical;
    line-height: 1.2;
    transition: color 0.3s;
}

.grid-item:active .grid-item-name {
    color: #ffffff !important;
}

.grid-item-owner {
    margin: 0 !important;
    font-size: 0.75rem;
    color: #666;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.grid-item-categories {
    margin: 5px 0 0 0 !important;
    font-size: 0.7rem;
    color: #3c8f44;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Style pour les catégories dans la vue liste */
.farmer-categories {
    font-size: 0.75rem;
    color: #3c8f44;
    margin: 0 !important;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

/* Styles pour le message d'info des distances */
.distances-info {
    text-align: center;
    margin: 10px 0 15px;
    padding: 8px;
    background-color: rgba(240, 249, 255, 0.85);
    border-radius: 5px;
    color: #0c5460;
    font-size: 0.9rem;
    backdrop-filter: blur(3px);
}

/* Styles pour le bouton de géolocalisation */
.geolocation-button {
    display: block;
    width: 100%;
    max-width: 300px;
    margin: 0 auto 15px;
    padding: 10px 15px;
    background-color: #3c8f44;
    color: white;
    border: none;
    border-radius: 4px;
    cursor: pointer;
    font-size: 0.95rem;
    transition: background-color 0.2s;
}

.geolocation-button:hover {
    background-color: #357a3b;
}

.geolocation-button:disabled {
    background-color: #8abb8e;
    cursor: not-allowed;
}

/* Styles pour la vue liste (existants) */
.farmer-list {
    padding: 0;
    margin: 0;
    list-style: none;
}

.farmer-item {
    display: block;
    background-color: rgba(255, 255, 255, 0.85);
    margin-bottom: 10px;
    border-radius: 5px;
    overflow: hidden;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    text-decoration: none !important;
    color: inherit !important;
    transition: transform 0.2s, box-shadow 0.2s;
    backdrop-filter: blur(3px);
}

.farmer-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}

.farmer-content {
    display: flex;
    align-items: center;
    height: 80px;
}

.farmer-image {
    width: 80px;
    height: 80px;
    object-fit: cover;
    object-position: center;
    flex-shrink: 0;
}

.farmer-info {
    flex-grow: 1;
    padding: 10px;
    overflow: hidden;
}

.farmer-name {
    font-weight: bold !important;
    color: #3c8f44 !important;
    margin: 0 0 5px 0 !important;
    font-size: 1rem !important;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.farmer-owner {
    margin: 0 !important;
    font-size: 0.8rem;
    color: #666;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.farmer-distance {
    padding: 10px 15px;
    text-align: center;
    background-color: #e9f5dc;
    min-width: 70px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    flex-shrink: 0;
}

.distance-value {
    font-weight: bold;
    color: #333;
    font-size: 1.1rem;
    margin: 0 !important;
}

.distance-unit {
    font-size: 0.75rem;
    color: #666;
    margin: 0 !important;
}

/* Réponses de succès */
.geolocation-success {
    background-color: rgba(212, 237, 218, 0.9);
    color: #155724;
    padding: 8px;
    border-radius: 4px;
    margin: 10px auto;
    text-align: center;
    max-width: 300px;
    font-size: 0.9rem;
    border: 1px solid #c3e6cb;
    animation: fadeOut 3s forwards;
    position: fixed;
    bottom: 20px;
    left: 50%;
    transform: translateX(-50%);
    z-index: 1000;
    backdrop-filter: blur(3px);
}

@keyframes fadeOut {
    0% { opacity: 1; }
    80% { opacity: 1; }
    100% { opacity: 0; }
}

/* Pagination */
.pagination-container {
    margin-top: 20px;
    display: flex;
    flex-direction: column;
    align-items: center;
}

.pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    flex-wrap: wrap;
    gap: 5px;
    margin-bottom: 10px;
}

.page-item {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    min-width: 30px;
    height: 30px;
    padding: 0 5px;
    background-color: rgba(248, 249, 250, 0.85);
    border: 1px solid #ddd;
    border-radius: 4px;
    color: #3c8f44;
    text-decoration: none;
    font-size: 0.9rem;
    transition: all 0.2s ease;
    backdrop-filter: blur(3px);
}

.page-item:hover {
    background-color: rgba(233, 236, 239, 0.9);
    border-color: #ced4da;
}

.page-item.active {
    background-color: #3c8f44;
    border-color: #3c8f44;
    color: white;
    font-weight: bold;
}

.page-item.ellipsis {
    border: none;
    background: none;
    color: #666;
    cursor: default;
}

.page-item.prev, .page-item.next, .page-item.first {
    font-size: 0.8rem;
}

.page-info {
    font-size: 0.85rem;
    color: white;
    text-align: center;
    text-shadow: 0 1px 2px rgba(0, 0, 0, 0.5);
}

/* Responsive */
@media (max-width: 767px) {
    .farmer-grid {
        grid-template-columns: repeat(2, 1fr);
    }
}

@media (max-width: 576px) {
    .farmer-content {
        height: 70px;
    }
    
    .farmer-image {
        width: 70px;
        height: 70px;
    }
    
    .farmer-distance {
        min-width: 60px;
        padding: 8px 10px;
    }
    
    .distance-value {
        font-size: 1rem;
    }
    
    .distance-unit {
        font-size: 0.7rem;
    }
    
    .farmer-name {
        font-size: 0.9rem !important;
    }
    
    .farmer-owner {
        font-size: 0.75rem;
    }
    
    .page-item {
        min-width: 26px;
        height: 26px;
        font-size: 0.8rem;
    }
    
    .grid-item-name {
        font-size: 0.8rem !important;
    }
    
    .grid-item-owner {
        font-size: 0.7rem;
    }
}
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const moduleId = '<?php echo $module->id; ?>';
    const geolocateButton = document.getElementById('geolocate-button-' + moduleId);
    const distancesInfo = document.getElementById('distances-info-' + moduleId);
    const root = '<?php echo $root; ?>';
    // Utilisation d'une variable globale pour l'URL de base des profils
    const profileUrlBase = '<?php echo $profileUrlBase; ?>';
    const currentPage = <?php echo $currentPage; ?>;
    
    // Références aux boutons de navigation
    const mapViewButton = document.getElementById('map-view-button');
    const gridViewButton = document.getElementById('grid-view-button');
    const searchButton = document.getElementById('search-button');
    
    // Références aux conteneurs de vues
    const mapContainer = document.getElementById('map-container');
    const gridContainer = document.getElementById('grid-container');
    
    // Références aux vues liste et grille
    const listView = document.getElementById('list-view');
    const gridView = document.getElementById('grid-view');
    
    // Références au popup de recherche et aux éléments associés
    const searchPopup = document.getElementById('search-popup');
    const searchInput = document.getElementById('search-input');
    const searchSubmit = document.getElementById('search-submit');
    const searchClose = document.getElementById('search-close');
    const searchResultsContainer = document.getElementById('search-results-container');
    const searchResultsList = document.getElementById('search-results-list');
    const searchCount = document.getElementById('search-count');
    
    // Variables pour la vue active
    const activeView = '<?php echo $activeView; ?>';
    
    // Cache pour l'index de recherche
    let searchIndex = null;
    
    // Variables pour la carte Google Maps
    let map;
    let markers = [];
    let markerCluster;
    
    // Fonction pour stocker le mode de vue actuel en localStorage
    function setViewMode(mode) {
        try {
            localStorage.setItem('agriculteur_view_mode', mode);
        } catch (e) {
            console.error('Impossible de stocker le mode de vue:', e);
        }
        
        // Ajouter le paramètre à tous les liens de pagination
        updatePaginationLinks(mode);
    }
    
    // Fonction pour mettre à jour les liens de pagination avec le mode de vue actif
    function updatePaginationLinks(viewMode) {
        document.querySelectorAll('.pagination a').forEach(function(link) {
            const url = new URL(link.href);
            url.searchParams.set('view_mode', viewMode);
            link.href = url.toString();
        });
    }
    
    // Ajoutons un bouton de vue liste (qui n'est pas dans la maquette mais utile pour l'UX)
    const listViewButton = document.createElement('button');
    listViewButton.innerHTML = '<i class="fas fa-list"></i>';
    listViewButton.className = 'nav-button list-view-button';
    listViewButton.style.marginRight = '0px';
    
    // Insérer avant le bouton de grille
    if (gridViewButton && gridViewButton.parentNode) {
        gridViewButton.parentNode.insertBefore(listViewButton, gridViewButton);
    }
    
    // Marquer le bouton liste comme actif si c'est la vue actuelle
    if (activeView === 'list') {
        listViewButton.classList.add('active');
    }
    
    // Gestionnaires d'événements pour les boutons de navigation
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            // Activer le bouton carte
            mapViewButton.classList.add('active');
            gridViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            
            // Afficher la carte et masquer la grille
            mapContainer.style.display = 'block';
            gridContainer.style.display = 'none';
            
            // Initialiser la carte si elle n'existe pas encore
            if (!map) {
                loadGoogleMapsWithClusterer();
            } else {
                // Si la carte existe déjà, rafraîchir son affichage
                google.maps.event.trigger(map, 'resize');
            }
            
            // Stocker le mode "map" (bien que ce ne soit pas utilisé dans la pagination)
            setViewMode('map');
        });
    }
    
    if (gridViewButton) {
        gridViewButton.addEventListener('click', function() {
            // Activer le bouton grille
            gridViewButton.classList.add('active');
            mapViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            
            // Afficher la grille et masquer la carte
            gridContainer.style.display = 'block';
            mapContainer.style.display = 'none';
            
            // Afficher la vue grille et masquer la vue liste
            gridView.classList.add('active-view');
            listView.classList.remove('active-view');
            searchResultsContainer.style.display = 'none';
            
            // Stocker le mode "grid"
            setViewMode('grid');
        });
    }
    
    // Ajouter l'événement pour basculer vers la vue liste
    listViewButton.addEventListener('click', function() {
        // Si la carte est visible, basculer d'abord vers le conteneur de grille
        if (mapContainer.style.display !== 'none') {
            mapViewButton.classList.remove('active');
            gridViewButton.classList.remove('active');
            mapContainer.style.display = 'none';
            gridContainer.style.display = 'block';
        }
        
        // Activer la vue liste
        listView.classList.add('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'none';
        
        // Mettre à jour l'apparence des boutons
        listViewButton.classList.add('active');
        gridViewButton.classList.remove('active');
        
        // Stocker le mode "list"
        setViewMode('list');
    });
    
    // Gestionnaire pour le bouton de recherche
    if (searchButton) {
        searchButton.addEventListener('click', function() {
            searchPopup.style.display = 'flex';
            searchInput.focus();
            searchInput.value = ''; // Réinitialise le champ de recherche
            
            // Charger l'index de recherche s'il n'est pas déjà chargé
            if (!searchIndex) {
                loadSearchIndex();
            }
        });
    }
    
    // Fermer le popup lorsqu'on clique sur le bouton fermer
    if (searchClose) {
        searchClose.addEventListener('click', function() {
            searchPopup.style.display = 'none';
        });
    }
    
    // Fermer le popup en cliquant en dehors de celui-ci
    searchPopup.addEventListener('click', function(event) {
        if (event.target === searchPopup) {
            searchPopup.style.display = 'none';
        }
    });
    
    // Soumettre la recherche
    if (searchSubmit) {
        searchSubmit.addEventListener('click', function() {
            performSearch();
        });
    }
    
    // Soumettre aussi lors de l'appui sur Entrée
    if (searchInput) {
        searchInput.addEventListener('keypress', function(event) {
            if (event.key === 'Enter') {
                performSearch();
            }
        });
    }
    
    // Fonction pour charger l'index de recherche
    function loadSearchIndex() {
        console.log('Chargement de l\'index de recherche...');
        // Essayer de charger l'index JSON
        fetch('/media/mod_agriculteur_liste/search-index.json')
            .then(response => {
                if (!response.ok) {
                    throw new Error(`Erreur HTTP: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data && Array.isArray(data.entries)) {
                    searchIndex = data.entries;
                    console.log('Index de recherche chargé avec ' + searchIndex.length + ' entrées');
                } else {
                    console.warn('Format d\'index invalide:', data);
                    loadIndexFromDOM();
                }
            })
            .catch(error => {
                console.error('Erreur lors du chargement de l\'index:', error);
                // Fallback: charger à partir du DOM
                loadIndexFromDOM();
            });
    }
    
    // Fonction de secours pour charger l'index à partir du DOM
    function loadIndexFromDOM() {
        console.log('Chargement de l\'index à partir du DOM...');
        const farmerItems = document.querySelectorAll('.farmer-item, .grid-item');
        searchIndex = [];
        
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const coordinates = item.getAttribute('data-coordinates');
            
            // Rechercher tous les attributs data-* pour enrichir l'index
            const dataAttributes = {};
            for (let attr of item.attributes) {
                if (attr.name.startsWith('data-') && attr.name !== 'data-id' && 
                    attr.name !== 'data-name' && attr.name !== 'data-owner') {
                    const key = attr.name.replace('data-', '');
                    dataAttributes[key] = attr.value;
                }
            }
            
            // Rechercher du texte supplémentaire dans les éléments enfants
            const additionalText = Array.from(item.querySelectorAll('p, h3, span, div'))
                .map(el => el.textContent.trim())
                .filter(text => text.length > 0)
                .join(' ');
            
            // Créer un texte de recherche enrichi
            const searchableText = [
                name || '',
                owner || '',
                additionalText,
                ...Object.values(dataAttributes)
            ].join(' ').toLowerCase();
            
            // Récupérer l'URL de l'image
            let image = '';
            const imgElement = item.querySelector('img');
            if (imgElement && imgElement.src) {
                image = imgElement.src;
            }
            
            searchIndex.push({
                id: id,
                name: name || '',
                owner: owner || '',
                coordinates: coordinates || '',
                searchableText: searchableText,
                image: image,
                element: item
            });
        });
        
        console.log('Index de secours chargé avec ' + searchIndex.length + ' entrées');
    }
    
    // Fonction pour effectuer la recherche
    function performSearch() {
        const query = searchInput.value.trim().toLowerCase();
        
        if (!query) {
            alert('Veuillez entrer un terme de recherche');
            return;
        }
        
        if (!searchIndex || searchIndex.length === 0) {
            console.log('Index non chargé, tentative de chargement...');
            loadSearchIndex();
            // Attendre un peu que l'index soit chargé
            setTimeout(() => performSearch(), 500);
            return;
        }
        
        console.log('Recherche de: "' + query + '" dans ' + searchIndex.length + ' entrées');
        
        // Rechercher dans l'index
        let results = [];
        
        // Détecter si c'est un index complet ou un index de secours DOM
        if (searchIndex.length > 0 && searchIndex[0].element) {
            console.log('Recherche dans l\'index DOM');
            
            // Index de secours DOM - amélioré
            // Diviser la requête en termes individuels pour une recherche plus précise
            const terms = query.split(/\s+/).filter(term => term.length > 1);
            
            searchIndex.forEach(item => {
                let score = 0;
                const searchableText = item.searchableText || '';
                
                // Vérifier la correspondance exacte
                if (searchableText.includes(query)) {
                    score += 10; // Score élevé pour correspondance exacte
                }
                
                // Vérifier chaque terme individuellement
                if (terms.length > 0) {
                    let termMatches = 0;
                    
                    terms.forEach(term => {
                        if (searchableText.includes(term)) {
                            termMatches++;
                            
                            // Bonus si le terme est dans le nom ou propriétaire
                            if (item.name.toLowerCase().includes(term)) {
                                score += 3;
                            }
                            if (item.owner.toLowerCase().includes(term)) {
                                score += 2;
                            }
                        }
                    });
                    
                    // Score basé sur le pourcentage de termes correspondants
                    if (termMatches > 0) {
                        score += (termMatches / terms.length) * 5;
                    }
                }
                
                if (score > 0) {
                    results.push({
                        item: item,
                        score: score
                    });
                }
            });
            
        } else {
            console.log('Recherche dans l\'index JSON');
            
            // Index complet JSON - amélioré
            // Pour les recherches à plusieurs mots
            const terms = query.split(/\s+/).filter(term => term.length > 1);
            
            searchIndex.forEach(item => {
                let score = 0;
                const searchableText = item.searchableText || '';
                
                // Recherche exacte dans le texte complet
                if (searchableText && searchableText.includes(query)) {
                    score += 10;
                }
                
                // Recherche par termes individuels avec scoring amélioré
                if (terms.length > 0) {
                    let termMatches = 0;
                    
                    terms.forEach(term => {
                        if (searchableText && searchableText.includes(term)) {
                            termMatches++;
                            
                            // Bonus pour des correspondances dans des champs spécifiques
                            if (item.name && item.name.toLowerCase().includes(term)) {
                                score += 3;
                            }
                            if (item.owner && item.owner.toLowerCase().includes(term)) {
                                score += 2;
                            }
                            if (item.description && item.description.toLowerCase().includes(term)) {
                                score += 1;
                            }
                            
                            // Bonus pour les correspondances dans d'autres champs
                            ['categories', 'products', 'labels', 'certifications', 'sales_methods'].forEach(field => {
                                if (Array.isArray(item[field])) {
                                    item[field].forEach(value => {
                                        if (typeof value === 'string' && value.toLowerCase().includes(term)) {
                                            score += 1;
                                        }
                                    });
                                }
                            });
                        }
                    });
                    
                    // Score basé sur le pourcentage de termes correspondants
                    if (termMatches > 0) {
                        score += (termMatches / terms.length) * 5;
                    }
                }
                
                if (score > 0) {
                    results.push({
                        item: item,
                        score: score
                    });
                }
            });
        }
        
        // Trier les résultats par score
        results.sort((a, b) => b.score - a.score);
        
        // Extraire les items pour l'affichage
        results = results.map(result => result.item);
        
        console.log('Résultats trouvés: ' + results.length);
        
        // Fermer le popup
        searchPopup.style.display = 'none';
        
        // Afficher les résultats
        displaySearchResults(results, query);
    }
    
    // Fonction pour afficher les résultats
    function displaySearchResults(results, query) {
        console.log('Affichage des résultats pour: "' + query + '"');
        
        // Nettoyer la liste des résultats précédents
        searchResultsList.innerHTML = '';
        
        // Afficher le nombre de résultats
        if (results.length === 0) {
            searchCount.textContent = 'Aucun résultat trouvé pour "' + query + '"';
            searchResultsList.innerHTML = '<div class="no-results">Aucun résultat ne correspond à votre recherche. Essayez avec un autre terme.</div>';
        } else {
            searchCount.textContent = results.length + ' résultat(s) trouvé(s) pour "' + query + '"';
            
            // Ajouter chaque résultat à la liste
            results.forEach(function(result) {
                if (result.element) {
                    // Si c'est un élément DOM (index de secours)
                    const listItem = document.createElement('li');
                    const farmerItem = result.element.cloneNode(true);
                    listItem.appendChild(farmerItem);
                    searchResultsList.appendChild(listItem);
                } else {
                    // Si c'est un objet JSON (index complet)
                    const listItem = document.createElement('li');
                    
                    // S'assurer que l'ID est bien défini pour construire l'URL du profil
                    const submissionId = result.id || '';
                    
                    // Construire l'URL du profil avec le contexte
                    const profileUrl = profileUrlBase + '?submission_id=' + submissionId 
                        + '&page_retour=' + currentPage 
                        + '&view_mode=' + activeView;
                    
                    // Afficher les catégories si disponibles
                    let categoriesHtml = '';
                    if (result.categories && result.categories.length > 0) {
                        categoriesHtml = '<p class="farmer-categories">' + result.categories.join(', ') + '</p>';
                    }
                    
                    // S'assurer que l'image est bien définie
                    let imageUrl = result.image || '';
                    // Si l'image est un chemin relatif, ajouter la base URL
                    if (imageUrl && !imageUrl.startsWith('http') && !imageUrl.startsWith('/')) {
                        imageUrl = root + imageUrl;
                    }
                    // Si aucune image n'est définie, utiliser une image par défaut
                    if (!imageUrl) {
                        imageUrl = root + 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                    }
                    
                    // Formater la distance (utiliser 0 comme valeur par défaut)
                    const distance = result.distance || 0;
                    const formattedDistance = Number.isFinite(distance) ? distance.toFixed(1) : "-";
                    
                    // Créer l'élément HTML
                    listItem.innerHTML = `
                        <a href="${profileUrl}" class="farmer-item" 
                           data-id="${submissionId}"
                           data-name="${result.name || ''}"
                           data-owner="${result.owner || ''}"
                           data-coordinates="${result.coordinates || ''}">
                            <div class="farmer-content">
                                <img src="${imageUrl}" alt="${result.name || ''}" class="farmer-image" onerror="this.src='${root}media/mod_agriculteur_liste/images/default-farmer.jpg'">
                                <div class="farmer-info">
                                    <h3 class="farmer-name">${result.name || ''}</h3>
                                    <p class="farmer-owner">${result.owner || ''}</p>
                                    ${categoriesHtml}
                                </div>
                                <div class="farmer-distance" id="distance-search-${submissionId}">
                                    <p class="distance-value">${formattedDistance}</p>
                                    <p class="distance-unit">km</p>
                                </div>
                            </div>
                        </a>
                    `;
                    
                    searchResultsList.appendChild(listItem);
                }
            });
        }
        
        // Masquer les autres vues et afficher les résultats
        listView.classList.remove('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'block';
        
        // Désactiver tous les boutons de vue
        if (listViewButton) listViewButton.classList.remove('active');
        if (gridViewButton) gridViewButton.classList.remove('active');
        
        // Faire défiler vers le haut
        document.querySelector('.top-navigation-bar').scrollIntoView({ behavior: 'smooth' });
        
        // Si le bouton de géolocalisation existe, recalculer les distances pour les résultats
        if (geolocateButton && navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    const userLat = position.coords.latitude;
                    const userLng = position.coords.longitude;
                    
                    // Calculer les distances pour les résultats de recherche
                    results.forEach(function(result) {
                        if (result.coordinates) {
                            const [farmLat, farmLng] = result.coordinates.split(',').map(parseFloat);
                            const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                            
                            // Mettre à jour l'affichage de la distance
                            const distanceElement = document.getElementById('distance-search-' + result.id);
                            if (distanceElement) {
                                const distanceValue = distanceElement.querySelector('.distance-value');
                                if (distanceValue) {
                                    distanceValue.textContent = distance.toFixed(1);
                                }
                            }
                        }
                    });
                },
                function(error) {
                    console.error('Erreur de géolocalisation:', error.message);
                }
            );
        }
    }
    
    // Chargement de l'API Google Maps et du script de clustering
    function loadGoogleMapsWithClusterer() {
        // Vérifier si Google Maps est déjà chargé
        if (typeof google === 'undefined' || typeof google.maps === 'undefined') {
            // Charger d'abord l'API Google Maps
            const gMapScript = document.createElement('script');
            gMapScript.src = 'https://maps.googleapis.com/maps/api/js?key=AIzaSyAjsOoV0528RtrLFvylBrsypk0Z3qik5RI&callback=initGoogleMapAfterLoad';
            gMapScript.async = true;
            gMapScript.defer = true;
            
            // Puis charger MarkerClustererPlus
            const clusterScript = document.createElement('script');
            clusterScript.src = 'https://unpkg.com/@googlemaps/markerclustererplus/dist/index.min.js';
            
            // Définir une fonction de rappel globale pour initialiser la carte après le chargement
            window.initGoogleMapAfterLoad = function() {
                console.log('Google Maps API chargée');
                // Attendre que le script de clustering soit également chargé
                if (typeof MarkerClusterer === 'undefined') {
                    setTimeout(initGoogleMapAfterLoad, 100); // Vérifier toutes les 100ms
                } else {
                    initMap();
                }
            };
            
            // Ajouter les scripts dans le bon ordre
            document.head.appendChild(clusterScript);
            document.head.appendChild(gMapScript);
        } else {
            // Google Maps est déjà chargé
            if (typeof MarkerClusterer === 'undefined') {
                // Charger uniquement le script de clustering
                const clusterScript = document.createElement('script');
                clusterScript.src = 'https://unpkg.com/@googlemaps/markerclustererplus/dist/index.min.js';
                clusterScript.onload = function() {
                    initMap();
                };
                document.head.appendChild(clusterScript);
            } else {
                // Tout est déjà chargé
                initMap();
            }
        }
    }
    
    // Initialisation de la carte Google Maps
    function initMap() {
        // Coordonnées centrales de La Réunion
        const reunionCenter = { lat: -21.115141, lng: 55.536384 };
        
        // Créer la carte avec les contrôles de zoom activés et bien visibles
        map = new google.maps.Map(document.getElementById('google-map'), {
            center: reunionCenter,
            zoom: 10,
            mapTypeId: google.maps.MapTypeId.ROADMAP,
            mapTypeControl: true,
            mapTypeControlOptions: {
                style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
                position: google.maps.ControlPosition.TOP_RIGHT
            },
            zoomControl: true,
            zoomControlOptions: {
                position: google.maps.ControlPosition.RIGHT_CENTER
            },
            streetViewControl: true,
            streetViewControlOptions: {
                position: google.maps.ControlPosition.RIGHT_BOTTOM
            },
            fullscreenControl: true
        });
        
// Ajouter des marqueurs pour chaque agriculteur
        markers = [];
        const farmerItems = document.querySelectorAll('.farmer-item, .grid-item');
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            const coordinates = item.getAttribute('data-coordinates');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const profileUrl = item.href;
            
            if (coordinates) {
                const [lat, lng] = coordinates.split(',').map(parseFloat);
                
                // Créer un marqueur
                const marker = new google.maps.Marker({
                    position: { lat: lat, lng: lng },
                    map: map,
                    title: name,
                    icon: {
                        url: 'https://maps.google.com/mapfiles/ms/icons/green-dot.png'
                    }
                });
                
                // Créer un élément de contenu DOM au lieu d'une chaîne HTML
                // Cette approche contourne les restrictions CORS
                const infowindowContent = document.createElement('div');
                infowindowContent.className = 'custom-infowindow';
                
                const contentDiv = document.createElement('div');
                contentDiv.className = 'infowindow-content';
                
                const titleElement = document.createElement('h3');
                titleElement.className = 'infowindow-title';
                titleElement.textContent = name;
                
                const subtitleElement = document.createElement('p');
                subtitleElement.className = 'infowindow-subtitle';
                subtitleElement.textContent = owner;
                
                const linkElement = document.createElement('a');
                linkElement.className = 'infowindow-link';
                linkElement.href = profileUrl;
                linkElement.textContent = 'Voir le profil';
                linkElement.style.color = 'white';
                linkElement.style.backgroundColor = '#3c8f44';
                linkElement.style.padding = '5px 10px';
                linkElement.style.borderRadius = '3px';
                linkElement.style.textDecoration = 'none';
                linkElement.style.display = 'inline-block';
                linkElement.style.marginTop = '5px';
                
                // Si vous voulez tenter d'ajouter une image
                const imgContainer = document.createElement('div');
                imgContainer.className = 'infowindow-image-container';
                imgContainer.style.position = 'relative';
                imgContainer.style.width = '100%';
                imgContainer.style.paddingTop = '56.25%';
                imgContainer.style.overflow = 'hidden';
                
                const imgElement = document.createElement('img');
                imgElement.className = 'infowindow-image';
                imgElement.alt = name;
                imgElement.style.position = 'absolute';
                imgElement.style.top = '0';
                imgElement.style.left = '0';
                imgElement.style.width = '100%';
                imgElement.style.height = '100%';
                imgElement.style.objectFit = 'cover';
                
                // Trouver l'image dans l'item
                const existingImg = item.querySelector('img');
                if (existingImg && existingImg.src) {
                    imgElement.src = existingImg.src;
                }
                
                // Assembler les éléments
                imgContainer.appendChild(imgElement);
                contentDiv.appendChild(titleElement);
                contentDiv.appendChild(subtitleElement);
                contentDiv.appendChild(linkElement);
                infowindowContent.appendChild(imgContainer);
                infowindowContent.appendChild(contentDiv);
                
                // Créer l'infowindow avec le contenu DOM
                const infowindow = new google.maps.InfoWindow({
                    content: infowindowContent,
                    maxWidth: 300
                });
                
                // Ouvrir l'infowindow au clic
                marker.addListener('click', function() {
                    // Fermer tous les autres infowindows
                    markers.forEach(function(m) {
                        if (m.infowindow) {
                            m.infowindow.close();
                        }
                    });
                    
                    // Ouvrir l'infowindow de ce marqueur
                    infowindow.open(map, marker);
                });
                
                // Stocker le marqueur et son infowindow
                marker.infowindow = infowindow;
                markers.push(marker);
            }
        });
        
        // Initialiser le MarkerClusterer avec des options personnalisées
        if (markers.length > 0) {
            // Créer le marker clusterer
            markerCluster = new MarkerClusterer(map, markers, {
                imagePath: 'https://developers.google.com/maps/documentation/javascript/examples/markerclusterer/m',
                gridSize: 50,
                maxZoom: 15
            });
        }
    }
    
    // Ajouter un gestionnaire d'événement au bouton de géolocalisation
    if (geolocateButton) {
        geolocateButton.addEventListener('click', getUserLocation);
    }
    
    // Fonction pour obtenir la position de l'utilisateur
    function getUserLocation() {
        if (!navigator.geolocation) {
            alert("La géolocalisation n'est pas supportée par votre navigateur.");
            return;
        }
        
        // Désactiver le bouton pendant la géolocalisation
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Localisation en cours...';
            geolocateButton.disabled = true;
        }
        
        navigator.geolocation.getCurrentPosition(
            // Succès
            function(position) {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                console.log('Position utilisateur obtenue:', userLat, userLng);
                
                // Afficher un message d'info
                if (distancesInfo) {
                    distancesInfo.style.display = 'block';
                }
                
                // Calculer les distances vers tous les agriculteurs
                calculateDistances(userLat, userLng);
                
                // Si la carte est initialisée, y ajouter la position de l'utilisateur
                if (map) {
                    const userPosition = new google.maps.LatLng(userLat, userLng);
                    
                    // Ajouter un marqueur pour l'utilisateur
                    const userMarker = new google.maps.Marker({
                        position: userPosition,
                        map: map,
                        title: 'Votre position',
                        icon: {
                            url: 'https://maps.google.com/mapfiles/ms/icons/blue-dot.png'
                        }
                    });
                    
                    // Centrer la carte sur l'utilisateur
                    map.setCenter(userPosition);
                    map.setZoom(12);
                }
            },
            // Erreur
            function(error) {
                console.error('Erreur de géolocalisation:', error.message);
                
                let errorMessage = "Impossible d'obtenir votre position.";
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += " Vous avez refusé l'accès à votre position.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += " Les informations de position ne sont pas disponibles.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += " La demande de position a expiré.";
                        break;
                    case error.UNKNOWN_ERROR:
                        errorMessage += " Une erreur inconnue s'est produite.";
                        break;
                }
                
                alert(errorMessage);
                
                // Réinitialiser le bouton
                if (geolocateButton) {
                    geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
                    geolocateButton.disabled = false;
                }
            },
            // Options
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            }
        );
    }
    
    // Fonction pour calculer les distances vers tous les agriculteurs
    function calculateDistances(userLat, userLng) {
        // Obtenir tous les éléments d'agriculteurs (liste et grille)
        const farmerItems = document.querySelectorAll('.farmer-item');
        const gridItems = document.querySelectorAll('.grid-item');
        
        // Calculer les distances à vol d'oiseau pour chaque agriculteur
        let distances = [];
        
        farmerItems.forEach(function(item) {
            const coordinates = item.getAttribute('data-coordinates');
            const id = item.getAttribute('data-id');
            
            if (coordinates) {
                const [farmLat, farmLng] = coordinates.split(',').map(parseFloat);
                const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                
                distances.push({
                    id: id,
                    distance: distance,
                    element: item
                });
                
                // Mettre à jour l'affichage dans la vue liste
                updateDistanceDisplay('distance-' + id, distance);
                
                // Mettre à jour l'affichage dans la vue grille
                updateDistanceDisplay('grid-distance-' + id, distance);
                
                // Stocker la distance dans l'élément pour le tri
                item.setAttribute('data-distance', distance);
                
                // Faire la même chose pour l'élément correspondant dans la vue grille
                const gridItem = document.querySelector(`.grid-item[data-id="${id}"]`);
                if (gridItem) {
                    gridItem.setAttribute('data-distance', distance);
                }
            }
        });
        
        // Trier les agriculteurs par distance
        distances.sort((a, b) => a.distance - b.distance);
        
        // Réorganiser les éléments dans le DOM (vue liste)
        const listContainer = document.getElementById('farmer-list-' + moduleId);
        if (listContainer) {
            const sortedItems = distances.map(item => item.element.parentNode);
            sortedItems.forEach(li => {
                if (li) listContainer.appendChild(li);
            });
        }
        
        // Réorganiser les éléments dans le DOM (vue grille)
        const gridContainer = document.getElementById('farmer-grid-' + moduleId);
        if (gridContainer) {
            const gridItems = Array.from(document.querySelectorAll('.grid-item')).sort((a, b) => {
                return parseFloat(a.getAttribute('data-distance')) - parseFloat(b.getAttribute('data-distance'));
            });
            gridItems.forEach(item => {
                gridContainer.appendChild(item);
            });
        }
        
        // Réinitialiser le bouton
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            geolocateButton.disabled = false;
        }
        
        // Afficher un message de succès
        showSuccessMessage();
    }
    
    // Fonction pour mettre à jour l'affichage de la distance
    function updateDistanceDisplay(elementId, distance) {
        const distanceElement = document.getElementById(elementId);
        if (!distanceElement) return;
        
        const distanceValue = distanceElement.querySelector('.distance-value') || 
                              distanceElement.querySelector('.grid-distance-value');
        
        if (distanceValue) {
            distanceValue.textContent = distance.toFixed(1);
        }
    }
    
    // Fonction pour calculer la distance à vol d'oiseau (formule Haversine)
    function calculateHaversineDistance(lat1, lng1, lat2, lng2) {
        const R = 6371; // Rayon de la Terre en km
        const dLat = toRad(lat2 - lat1);
        const dLng = toRad(lng2 - lng1);
        
        const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) *
                Math.sin(dLng / 2) * Math.sin(dLng / 2);
                
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c;
    }
    
    // Fonction pour convertir degrés en radians
    function toRad(degrees) {
        return degrees * Math.PI / 180;
    }
    
    // Fonction pour afficher un message de succès temporaire
    function showSuccessMessage() {
        const successMessage = document.createElement('div');
        successMessage.className = 'geolocation-success';
        successMessage.innerHTML = '<i class="fas fa-check-circle"></i> Distances calculées avec succès !';
        
        document.body.appendChild(successMessage);
        
        // Supprimer le message après quelques secondes
        setTimeout(function() {
            successMessage.remove();
        }, 3000);
    }
    
    // Charger Google Maps et le script de clustering lorsque le bouton de carte est cliqué
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            if (!map) {
                loadGoogleMapsWithClusterer();
            }
        });
    }
    
    // Précharger l'index de recherche après le chargement initial de la page
    // pour que la recherche soit réactive dès la première utilisation
    setTimeout(function() {
        if (!searchIndex) {
            loadSearchIndex();
        }
    }, 2000); // Délai de 2 secondes pour ne pas ralentir le chargement initial
    
    // Mettre à jour les liens de pagination au chargement de la page
    updatePaginationLinks('<?php echo $activeView; ?>');
});
</script>