<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_profil
 *
 * @copyright   Copyright (C) 2025 Votre Nom. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;

/**
 * Helper pour le module Profil Agriculteur
 */
class ModAgriculteurProfilHelper
{
    /**
     * Récupère les données de profil par ID de formulaire
     *
     * @param   integer  $formId  L'ID du formulaire
     * @return  array|false  Les données du profil ou false si non trouvé
     */
    public function getProfileDataByFormId($formId)
    {
        // Accès à la base de données
        $db = Factory::getDbo();

        // Récupérer le dernier enregistrement soumis pour ce formulaire 
        $query = $db->getQuery(true)
            ->select('*')
            ->from($db->quoteName('#__rsform_submissions', 's'))
            ->where($db->quoteName('s.FormId') . ' = ' . $db->quote($formId))
            ->order($db->quoteName('s.DateSubmitted') . ' DESC');

        $db->setQuery($query, 0, 1);
        $submission = $db->loadObject();

        if (!$submission) {
            return false;
        }

        return $this->getProfileDataBySubmissionId($submission->SubmissionId);
    }

    /**
     * Récupère les données de profil par ID de soumission
     *
     * @param   integer  $submissionId  L'ID de la soumission
     * @return  array|false  Les données du profil ou false si non trouvé
     */
    public function getProfileDataBySubmissionId($submissionId)
    {
        // Accès à la base de données
        $db = Factory::getDbo();

        // Récupérer les valeurs des champs du formulaire pour cette soumission
        $query = $db->getQuery(true)
            ->select('sv.FieldName, sv.FieldValue')
            ->from($db->quoteName('#__rsform_submission_values', 'sv'))
            ->where($db->quoteName('sv.SubmissionId') . ' = ' . $db->quote($submissionId));

        $db->setQuery($query);
        $fields = $db->loadObjectList();

        if (empty($fields)) {
            return false;
        }

        // Formater les données pour une utilisation facile
        $profileData = [];
        foreach ($fields as $field) {
            $profileData[$field->FieldName] = $field->FieldValue;
        }

        // Stocker l'ID de soumission pour l'utiliser dans les URLs d'images si nécessaire
        $profileData['_submission_id'] = $submissionId;

        // Préparer les données pour le template
        return $this->prepareProfileData($profileData);
    }

    /**
     * Prépare les données du profil pour l'affichage
     *
     * @param   array  $rawData  Données brutes du formulaire
     * @return  array  Données formatées pour l'affichage
     */
    private function prepareProfileData($rawData)
    {
        // Récupérer l'ID de soumission pour les images
        $submissionId = isset($rawData['_submission_id']) ? $rawData['_submission_id'] : 0;
        
        // Créer une structure organisée pour les données du profil
        $profileData = [
            // ID de soumission pour les références d'images
            '_submission_id' => $submissionId,
            
            // Informations de base
            'societe' => $rawData['societe'] ?? '',
            'nom' => $rawData['nom'] ?? '',
            'prenom' => $rawData['prenom'] ?? '',
            // Utiliser le champ 'information' pour la section "À propos"
            'description' => $rawData['information'] ?? '',
            'inscription' => $rawData['inscription'] ?? '',
            'email' => $rawData['Email'] ?? '',
            'telephone' => $rawData['telephone'] ?? '',
            'Site-internet' => $rawData['Site-internet'] ?? '',
            'facebook' => $rawData['facebook'] ?? '',
            'adresse' => $rawData['adresse'] ?? '',
            'code_postal' => $rawData['code-postal'] ?? '',
            'ville' => $rawData['ville'] ?? '',
            
            // Coordonnées GPS pour la carte
            'geolocalisation' => $rawData['geolocalisation'] ?? '',
            
            // Statut et catégories
            'status' => $this->getSelectedRadioValue($rawData, 'status'),
            'autre_status' => $rawData['autre-status'] ?? '',
            
            // Catégories d'activité (checkboxes)
            'categories' => $this->getSelectedCheckboxes($rawData, 'categories'),
            
            // Types d'agriculture et d'élevage
            'type_agriculture' => $this->getSelectedCheckboxes($rawData, 'type-agriculture'),
            'type_elevage' => $this->getSelectedCheckboxes($rawData, 'type-elevage'),
            
            // Cultures et produits
            'cultures' => $this->getSelectedCheckboxes($rawData, 'cultures'),
            'legumes_tropicaux' => $this->getSelectedCheckboxes($rawData, 'legumes-tropicaux'),
            'legumes_classiques' => $this->getSelectedCheckboxes($rawData, 'legumes-classiques'),
            'fruits_tropicaux' => $this->getSelectedCheckboxes($rawData, 'fruits-tropicaux'),
            'agrumes' => $this->getSelectedCheckboxes($rawData, 'agrumes'),
            'fruits_specifiques' => $this->getSelectedCheckboxes($rawData, 'fruits-specifiques'),
            'cafe_vanille_cacao' => $this->getSelectedCheckboxes($rawData, 'choix-cafe-vanille-cacao'),
            'cultures_fourrageres' => $this->getSelectedCheckboxes($rawData, 'choix-cultures-fourrageres'),
            'plantes_aromatiques' => $this->getSelectedCheckboxes($rawData, 'Choix-plantes-aromatiques-et-medicinales'),
            'fleurs_horticulture' => $this->getSelectedCheckboxes($rawData, 'choix-cultures-fleurs-et-horticulture'),
            'plantes_fibres_epices' => $this->getSelectedCheckboxes($rawData, 'choix-plantes-fibres-epices'),
            
            // Types d'élevage spécifiques
            'elevage_bovin' => $this->getSelectedCheckboxes($rawData, 'elevage-bovin'),
            'elevage_volailles' => $this->getSelectedCheckboxes($rawData, 'elevage-volailles'),
            
            // Autres activités et services
            'autres_activites' => $this->getSelectedCheckboxes($rawData, 'autre'),
            'stages_formation' => $this->getSelectedCheckboxes($rawData, 'stage-formation'),
            
            // Produits et horaires
            'produits_semaine' => $rawData['produits-semaine'] ?? '',
            'jours_reception' => $this->getSelectedCheckboxes($rawData, 'jours-reception'),
            'horaires_reception' => $rawData['horaires-reception'] ?? '',
            
            // Informations pour le marché
            'marches' => $rawData['marches'] ?? '',
            
            // Médias - les originaux et les URLs
            'logo' => $rawData['logo'] ?? '',
            'logo_url' => $rawData['logo_url'] ?? '',
            'photo_couverture' => $rawData['photo-couverture'] ?? '',
            'photo_couverture_url' => $rawData['photo_couverture_url'] ?? '',
            'galerie_photos' => $rawData['galerie-photos_'] ?? '',
            'galerie_photos_urls' => $rawData['galerie_photos_urls'] ?? '',
            'video_v' => $rawData['video-v'] ?? '',
            'video_i' => $rawData['video-i'] ?? '',
            
            // Champ PHP personnalisé
            'php_code' => $rawData['login'] ?? ''
        ];
        
        // Fusionner des catégories pour faciliter l'affichage
        $profileData['all_legumes'] = array_merge(
            $profileData['legumes_tropicaux'] ?? [],
            $profileData['legumes_classiques'] ?? []
        );

        $profileData['all_fruits'] = array_merge(
            $profileData['fruits_tropicaux'] ?? [],
            $profileData['agrumes'] ?? [],
            $profileData['fruits_specifiques'] ?? []
        );

        $profileData['all_plantes'] = array_merge(
            $profileData['plantes_aromatiques'] ?? [],
            $profileData['plantes_fibres_epices'] ?? []
        );
        
        // Construire l'adresse complète
        $profileData['full_address'] = trim($profileData['adresse'] . ' ' . $profileData['code_postal'] . ' ' . $profileData['ville']);
        
        return $profileData;
    }

    /**
     * Récupère la valeur d'un bouton radio sélectionné
     *
     * @param   array   $fields     Tableau associatif des champs du formulaire
     * @param   string  $fieldName  Nom du champ
     * @return  string  Valeur sélectionnée
     */
    private function getSelectedRadioValue($fields, $fieldName)
    {
        // Pour les boutons radio, récupérer la valeur sélectionnée
        return $fields[$fieldName] ?? '';
    }

    /**
     * Récupère les valeurs des cases à cocher sélectionnées
     *
     * @param   array   $fields     Tableau associatif des champs du formulaire
     * @param   string  $fieldName  Nom du champ
     * @return  array   Liste des options sélectionnées
     */
    private function getSelectedCheckboxes($fields, $fieldName)
    {
        // Pour les cases à cocher, créer un tableau des options sélectionnées
        $result = [];
        
        // Si le champ n'existe pas ou est vide
        if (!isset($fields[$fieldName]) || empty($fields[$fieldName])) {
            return $result;
        }
        
        $fieldValue = $fields[$fieldName];
        
        // Tentative de décodage JSON d'abord
        $jsonData = json_decode($fieldValue, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($jsonData)) {
            // C'est un JSON valide
            return array_filter($jsonData, function($value) {
                return !empty($value);
            });
        }
        
        // Ce n'est pas du JSON, traiter comme une chaîne séparée par des virgules
        $result = explode(',', $fieldValue);
        // Nettoyer les valeurs
        $result = array_map('trim', $result);
        // Filtrer les valeurs vides
        $result = array_filter($result, function($value) {
            return !empty($value);
        });
        
        return $result;
    }

    /**
     * Génère un lien Google Maps avec itinéraires à partir des coordonnées
     *
     * @param   string  $coordinates  Coordonnées au format "latitude,longitude"
     * @return  string  URL Google Maps avec itinéraires
     */
    public function generateGoogleMapsLink($coordinates)
    {
        if (empty($coordinates)) {
            return '#';
        }
        // Utiliser l'API directioner pour avoir les itinéraires
        return 'https://www.google.com/maps/dir/?api=1&destination=' . urlencode($coordinates);
    }

    /**
     * Formate une URL en ajoutant http:// si nécessaire
     *
     * @param   string  $url  URL à formater
     * @return  string  URL formatée
     */
    public function formatUrl($url)
    {
        if (empty($url)) {
            return '';
        }
        
        if (!preg_match('~^(?:f|ht)tps?://~i', $url)) {
            $url = 'http://' . $url;
        }
        
        return $url;
    }

    /**
     * Obtient l'URL d'un fichier téléchargé
     *
     * @param   string  $filename  Nom du fichier
     * @return  string  URL du fichier
     */
    public function getUploadPath($filename)
    {
        if (empty($filename)) {
            return '';
        }
        
        // Si le chemin contient déjà le chemin du serveur, on l'extrait
        if (strpos($filename, '/home/') !== false) {
            // Extraire juste le nom du fichier de l'URL complète
            $parts = explode('/', $filename);
            $filename = end($parts);
        }
        
        // Construction de l'URL sans aucun chemin serveur
        return 'https://labiolela.re/components/com_rsform/uploads/' . $filename;
    }

    /**
     * Extrait l'URL d'intégration pour les vidéos YouTube et Vimeo
     *
     * @param   string  $videoUrl  URL de la vidéo
     * @return  string  URL d'intégration
     */
    public function getVideoEmbedUrl($videoUrl)
    {
        if (empty($videoUrl)) {
            return '';
        }
        
        // YouTube
        if (preg_match('/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/', $videoUrl, $matches)) {
            return 'https://www.youtube.com/embed/' . $matches[1];
        }
        
        // Vimeo
        if (preg_match('/vimeo\.com\/(?:video\/|channels\/\w+\/|groups\/[^\/]+\/videos\/|album\/\d+\/video\/|)(\d+)(?:$|\/|\?)/', $videoUrl, $matches)) {
            return 'https://player.vimeo.com/video/' . $matches[1];
        }
        
        return $videoUrl;
    }
}