/**
 * Script principal du module agriculteur_liste
 * Ce fichier contient toutes les fonctionnalités essentielles du module
 */

document.addEventListener('DOMContentLoaded', function() {
    // Variables et éléments du DOM
    const geolocateButton = document.getElementById('geolocate-button-' + moduleId);
    const distancesInfo = document.getElementById('distances-info-' + moduleId);
    
    // Références aux boutons de navigation
    const mapViewButton = document.getElementById('map-view-button');
    const gridViewButton = document.getElementById('grid-view-button');
    const searchButton = document.getElementById('search-button');
    
    // Références aux conteneurs de vues
    const mapContainer = document.getElementById('map-container');
    const gridContainer = document.getElementById('grid-container');
    
    // Références aux vues liste et grille
    const listView = document.getElementById('list-view');
    const gridView = document.getElementById('grid-view');
    
    // Références au popup de recherche et aux éléments associés
    const searchPopup = document.getElementById('search-popup');
    const searchInput = document.getElementById('search-input');
    const searchSubmit = document.getElementById('search-submit');
    const searchClose = document.getElementById('search-close');
    const searchResultsContainer = document.getElementById('search-results-container');
    const searchResultsList = document.getElementById('search-results-list');
    const searchCount = document.getElementById('search-count');
    
    // Variables pour le mode recherche
    let isSearchMode = false;        // Indique si on est en mode recherche
    let currentSearchQuery = '';     // Terme de recherche actuel
    let searchResults = [];          // Résultats de recherche complets
    let searchCurrentPage = 1;       // Page actuelle des résultats de recherche
    const searchItemsPerPage = 10;   // Nombre de résultats par page
    
    // Cache pour l'index de recherche
    let searchIndex = null;
    
    // Variables pour la carte Google Maps
    let map;
    let markers = [];
    let markerCluster;
    
    // ====================
    // = FONCTIONS UTILITAIRES
    // ====================
    
    // Fonction pour stocker le mode de vue actuel en localStorage
    function setViewMode(mode) {
        try {
            localStorage.setItem('agriculteur_view_mode', mode);
        } catch (e) {
            console.error('Impossible de stocker le mode de vue:', e);
        }
        
        // Ajouter le paramètre à tous les liens de pagination
        updatePaginationLinks(mode);
    }
    
    // Fonction pour mettre à jour les liens de pagination avec le mode de vue actif
    function updatePaginationLinks(viewMode) {
        document.querySelectorAll('.pagination a').forEach(function(link) {
            const url = new URL(link.href);
            url.searchParams.set('view_mode', viewMode);
            
            // Si en mode recherche, ajouter un paramètre pour maintenir le mode recherche
            if (isSearchMode && currentSearchQuery) {
                url.searchParams.set('search_query', encodeURIComponent(currentSearchQuery));
            }
            
            link.href = url.toString();
        });
    }
    
    // Fonction pour convertir degrés en radians
    function toRad(degrees) {
        return degrees * Math.PI / 180;
    }
    
    // Fonction pour calculer la distance à vol d'oiseau (formule Haversine)
    function calculateHaversineDistance(lat1, lng1, lat2, lng2) {
        const R = 6371; // Rayon de la Terre en km
        const dLat = toRad(lat2 - lat1);
        const dLng = toRad(lng2 - lng1);
        
        const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) *
                Math.sin(dLng / 2) * Math.sin(dLng / 2);
                
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c;
    }
    
    // Fonction pour mettre à jour l'affichage de la distance
    function updateDistanceDisplay(elementId, distance) {
        const distanceElement = document.getElementById(elementId);
        if (!distanceElement) return;
        
        const distanceValue = distanceElement.querySelector('.distance-value') || 
                            distanceElement.querySelector('.grid-distance-value');
        
        if (distanceValue) {
            distanceValue.textContent = distance.toFixed(1);
        }
    }

    // Fonction pour afficher un message de succès temporaire
    function showSuccessMessage() {
        const successMessage = document.createElement('div');
        successMessage.className = 'geolocation-success';
        successMessage.innerHTML = '<i class="fas fa-check-circle"></i> Distances calculées avec succès !';
        
        document.body.appendChild(successMessage);
        
        // Supprimer le message après quelques secondes
        setTimeout(function() {
            successMessage.remove();
        }, 3000);
    }
    
    // ====================
    // = GESTION DES VUES
    // ====================
    
    // Ajoutons un bouton de vue liste (qui n'est pas dans la maquette mais utile pour l'UX)
    const listViewButton = document.createElement('button');
    listViewButton.innerHTML = '<i class="fas fa-list"></i>';
    listViewButton.className = 'nav-button list-view-button';
    listViewButton.style.marginRight = '0px';
    
    // Insérer avant le bouton de grille
    if (gridViewButton && gridViewButton.parentNode) {
        gridViewButton.parentNode.insertBefore(listViewButton, gridViewButton);
    }
    
    // Marquer le bouton liste comme actif si c'est la vue actuelle
    if (activeView === 'list') {
        listViewButton.classList.add('active');
    }
    
    // Vérifier si on vient d'une URL avec un paramètre de recherche
    function checkForSearchQueryInURL() {
        const urlParams = new URLSearchParams(window.location.search);
        const searchQuery = urlParams.get('search_query');
        
        if (searchQuery) {
            // Décoder le terme de recherche
            const decodedQuery = decodeURIComponent(searchQuery);
            currentSearchQuery = decodedQuery;
            
            // Mettre à jour le champ de recherche
            if (searchInput) {
                searchInput.value = decodedQuery;
            }
            
            // Charger l'index et effectuer la recherche
            if (!searchIndex) {
                loadSearchIndex();
                setTimeout(() => {
                    performSearch(decodedQuery);
                }, 500);
            } else {
                performSearch(decodedQuery);
            }
            
            return true;
        }
        
        return false;
    }
    
    // Gestionnaires d'événements pour les boutons de navigation
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            // Si on est en mode recherche, quitter ce mode
            isSearchMode = false;
            
            // Activer le bouton carte
            mapViewButton.classList.add('active');
            gridViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            
            // Afficher la carte et masquer la grille
            mapContainer.style.display = 'block';
            gridContainer.style.display = 'none';
            
            // Initialiser la carte si elle n'existe pas encore
            if (!map) {
                loadGoogleMapsWithClusterer();
            } else {
                // Si la carte existe déjà, rafraîchir son affichage
                google.maps.event.trigger(map, 'resize');
            }
            
            // Stocker le mode "map" (bien que ce ne soit pas utilisé dans la pagination)
            setViewMode('map');
        });
    }
    
    if (gridViewButton) {
        gridViewButton.addEventListener('click', function() {
            // Si on est en mode recherche, quitter ce mode
            isSearchMode = false;
            
            // Activer le bouton grille
            gridViewButton.classList.add('active');
            mapViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            
            // Afficher la grille et masquer la carte
            gridContainer.style.display = 'block';
            mapContainer.style.display = 'none';
            
            // Afficher la vue grille et masquer la vue liste
            gridView.classList.add('active-view');
            listView.classList.remove('active-view');
            searchResultsContainer.style.display = 'none';
            
            // Stocker le mode "grid"
            setViewMode('grid');
        });
    }
    
    // Ajouter l'événement pour basculer vers la vue liste
    listViewButton.addEventListener('click', function() {
        // Si on est en mode recherche, quitter ce mode
        isSearchMode = false;
        
        // Si la carte est visible, basculer d'abord vers le conteneur de grille
        if (mapContainer.style.display !== 'none') {
            mapViewButton.classList.remove('active');
            gridViewButton.classList.remove('active');
            mapContainer.style.display = 'none';
            gridContainer.style.display = 'block';
        }
        
        // Activer la vue liste
        listView.classList.add('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'none';
        
        // Mettre à jour l'apparence des boutons
        listViewButton.classList.add('active');
        gridViewButton.classList.remove('active');
        
        // Stocker le mode "list"
        setViewMode('list');
    });
    
    // ====================
    // = RECHERCHE
    // ====================
    
    // Gestionnaire pour le bouton de recherche
    if (searchButton) {
        searchButton.addEventListener('click', function() {
            searchPopup.style.display = 'flex';
            searchInput.focus();
            
            // Si on a déjà un terme de recherche, le réutiliser
            if (currentSearchQuery) {
                searchInput.value = currentSearchQuery;
            } else {
                searchInput.value = ''; // Réinitialise le champ de recherche
            }
            
            // Charger l'index de recherche s'il n'est pas déjà chargé
            if (!searchIndex) {
                loadSearchIndex();
            }
        });
    }
    
    // Fermer le popup lorsqu'on clique sur le bouton fermer
    if (searchClose) {
        searchClose.addEventListener('click', function() {
            searchPopup.style.display = 'none';
        });
    }
    
    // Fermer le popup en cliquant en dehors de celui-ci
    searchPopup.addEventListener('click', function(event) {
        if (event.target === searchPopup) {
            searchPopup.style.display = 'none';
        }
    });
    
    // Soumettre la recherche
    if (searchSubmit) {
        searchSubmit.addEventListener('click', function() {
            const query = searchInput.value.trim();
            if (query) {
                performSearch(query);
            } else {
                alert('Veuillez entrer un terme de recherche');
            }
        });
    }
    
    // Soumettre aussi lors de l'appui sur Entrée
    if (searchInput) {
        searchInput.addEventListener('keypress', function(event) {
            if (event.key === 'Enter') {
                const query = searchInput.value.trim();
                if (query) {
                    performSearch(query);
                } else {
                    alert('Veuillez entrer un terme de recherche');
                }
            }
        });
    }
    
    // Fonction pour charger l'index de recherche
    function loadSearchIndex() {
        console.log('Chargement de l\'index de recherche...');
        // Essayer de charger l'index JSON
        fetch('/media/mod_agriculteur_liste/search-index.json')
            .then(response => {
                if (!response.ok) {
                    throw new Error(`Erreur HTTP: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data && Array.isArray(data.entries)) {
                    searchIndex = data.entries;
                    console.log('Index de recherche chargé avec ' + searchIndex.length + ' entrées');
                    
                    // Vérifier si on doit effectuer une recherche depuis l'URL
                    checkForSearchQueryInURL();
                } else {
                    console.warn('Format d\'index invalide:', data);
                    loadIndexFromDOM();
                }
            })
            .catch(error => {
                console.error('Erreur lors du chargement de l\'index:', error);
                // Fallback: charger à partir du DOM
                loadIndexFromDOM();
            });
    }
    
    // Fonction de secours pour charger l'index à partir du DOM
    function loadIndexFromDOM() {
        console.log('Chargement de l\'index à partir du DOM...');
        const farmerItems = document.querySelectorAll('.farmer-item, .grid-item');
        searchIndex = [];
        
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const coordinates = item.getAttribute('data-coordinates');
            
            // Rechercher tous les attributs data-* pour enrichir l'index
            const dataAttributes = {};
            for (let attr of item.attributes) {
                if (attr.name.startsWith('data-') && attr.name !== 'data-id' && 
                    attr.name !== 'data-name' && attr.name !== 'data-owner') {
                    const key = attr.name.replace('data-', '');
                    dataAttributes[key] = attr.value;
                }
            }
            
            // Rechercher du texte supplémentaire dans les éléments enfants
            const additionalText = Array.from(item.querySelectorAll('p, h3, span, div'))
                .map(el => el.textContent.trim())
                .filter(text => text.length > 0)
                .join(' ');
            
            // Créer un texte de recherche enrichi
            const searchableText = [
                name || '',
                owner || '',
                additionalText,
                ...Object.values(dataAttributes)
            ].join(' ').toLowerCase();
            
            // Récupérer l'URL de l'image
            let image = '';
            const imgElement = item.querySelector('img');
            if (imgElement && imgElement.src) {
                image = imgElement.src;
            }
            
            searchIndex.push({
                id: id,
                name: name || '',
                owner: owner || '',
                coordinates: coordinates || '',
                searchableText: searchableText,
                image: image,
                element: item
            });
        });
        
        console.log('Index de secours chargé avec ' + searchIndex.length + ' entrées');
        
        // Vérifier si on doit effectuer une recherche depuis l'URL
        checkForSearchQueryInURL();
    }
    
    // Fonction pour effectuer la recherche
    function performSearch(query) {
        if (!query) {
            query = searchInput.value.trim().toLowerCase();
        } else {
            query = query.toLowerCase();
        }
        
        if (!query) {
            alert('Veuillez entrer un terme de recherche');
            return;
        }
        
        if (!searchIndex || searchIndex.length === 0) {
            console.log('Index non chargé, tentative de chargement...');
            loadSearchIndex();
            // Attendre un peu que l'index soit chargé
            setTimeout(() => performSearch(query), 500);
            return;
        }
        
        console.log('Recherche de: "' + query + '" dans ' + searchIndex.length + ' entrées');
        
        // Mettre à jour les variables du mode recherche
        isSearchMode = true;
        currentSearchQuery = query;
        searchCurrentPage = 1;
        
        // Rechercher dans l'index
        let results = [];
        
        // Détecter si c'est un index complet ou un index de secours DOM
        if (searchIndex.length > 0 && searchIndex[0].element) {
            console.log('Recherche dans l\'index DOM');
            
            // Index de secours DOM - amélioré
            // Diviser la requête en termes individuels pour une recherche plus précise
            const terms = query.split(/\s+/).filter(term => term.length > 1);
            
            searchIndex.forEach(item => {
                let score = 0;
                const searchableText = item.searchableText || '';
                
                // Vérifier la correspondance exacte
                if (searchableText.includes(query)) {
                    score += 10; // Score élevé pour correspondance exacte
                }
                
                // Vérifier chaque terme individuellement
                if (terms.length > 0) {
                    let termMatches = 0;
                    
                    terms.forEach(term => {
                        if (searchableText.includes(term)) {
                            termMatches++;
                            
                            // Bonus si le terme est dans le nom ou propriétaire
                            if (item.name.toLowerCase().includes(term)) {
                                score += 3;
                            }
                            if (item.owner.toLowerCase().includes(term)) {
                                score += 2;
                            }
                        }
                    });
                    
                    // Score basé sur le pourcentage de termes correspondants
                    if (termMatches > 0) {
                        score += (termMatches / terms.length) * 5;
                    }
                }
                
                if (score > 0) {
                    results.push({
                        item: item,
                        score: score
                    });
                }
            });
            
        } else {
            console.log('Recherche dans l\'index JSON');
            
            // Index complet JSON - amélioré
            // Pour les recherches à plusieurs mots
            const terms = query.split(/\s+/).filter(term => term.length > 1);
            
            searchIndex.forEach(item => {
                let score = 0;
                const searchableText = item.searchableText || '';
                
                // Recherche exacte dans le texte complet
                if (searchableText && searchableText.includes(query)) {
                    score += 10;
                }
                
                // Recherche par termes individuels avec scoring amélioré
                if (terms.length > 0) {
                    let termMatches = 0;
                    
                    terms.forEach(term => {
                        if (searchableText && searchableText.includes(term)) {
                            termMatches++;
                            
                            // Bonus pour des correspondances dans des champs spécifiques
                            if (item.name && item.name.toLowerCase().includes(term)) {
                                score += 3;
                            }
                            if (item.owner && item.owner.toLowerCase().includes(term)) {
                                score += 2;
                            }
                            if (item.description && item.description.toLowerCase().includes(term)) {
                                score += 1;
                            }
                            
                            // Bonus pour les correspondances dans d'autres champs
                            ['categories', 'products', 'labels', 'certifications', 'sales_methods'].forEach(field => {
                                if (Array.isArray(item[field])) {
                                    item[field].forEach(value => {
                                        if (typeof value === 'string' && value.toLowerCase().includes(term)) {
                                            score += 1;
                                        }
                                    });
                                }
                            });
                        }
                    });
                    
                    // Score basé sur le pourcentage de termes correspondants
                    if (termMatches > 0) {
                        score += (termMatches / terms.length) * 5;
                    }
                }
                
                if (score > 0) {
                    results.push({
                        item: item,
                        score: score
                    });
                }
            });
        }
        
        // Trier les résultats par score
        results.sort((a, b) => b.score - a.score);
        
        // Extraire les items pour l'affichage
        searchResults = results.map(result => result.item);
        
        console.log('Résultats trouvés: ' + searchResults.length);
        
        // Fermer le popup
        searchPopup.style.display = 'none';
        
        // Afficher les résultats
        displaySearchResults();
    }
    
    // Fonction pour afficher les résultats de recherche
    function displaySearchResults() {
        console.log('Affichage des résultats pour: "' + currentSearchQuery + '"');
        
        // Nettoyer la liste des résultats précédents
        searchResultsList.innerHTML = '';
        
        // Afficher le nombre de résultats
        if (searchResults.length === 0) {
            searchCount.textContent = 'Aucun résultat trouvé pour "' + currentSearchQuery + '"';
            searchResultsList.innerHTML = '<div class="no-results">Aucun résultat ne correspond à votre recherche. Essayez avec un autre terme.</div>';
            
            // Cacher la pagination car aucun résultat
            hideSearchPagination();
        } else {
            searchCount.textContent = searchResults.length + ' résultat(s) trouvé(s) pour "' + currentSearchQuery + '"';
            
            // Calculer les indices de début et de fin pour la pagination
            const startIndex = (searchCurrentPage - 1) * searchItemsPerPage;
            const endIndex = Math.min(startIndex + searchItemsPerPage, searchResults.length);
            
            // Extraire les résultats pour cette page
            const pageResults = searchResults.slice(startIndex, endIndex);
            
            // Ajouter chaque résultat à la liste
            pageResults.forEach(function(result) {
                if (result.element) {
                    // Si c'est un élément DOM (index de secours)
                    const listItem = document.createElement('li');
                    const farmerItem = result.element.cloneNode(true);
                    listItem.appendChild(farmerItem);
                    searchResultsList.appendChild(listItem);
                } else {
                    // Si c'est un objet JSON (index complet)
                    const listItem = document.createElement('li');
                    
                    // S'assurer que l'ID est bien défini pour construire l'URL du profil
                    const submissionId = result.id || '';
                    
                    // Construire l'URL du profil avec le contexte et maintenir le mode recherche
                    const profileUrl = profileUrlBase + '?submission_id=' + submissionId 
                        + '&page_retour=' + currentPage
                        + '&view_mode=' + activeView
                        + '&search_query=' + encodeURIComponent(currentSearchQuery)
                        + '&search_page=' + searchCurrentPage;
                    
                    // Afficher les catégories si disponibles
                    let categoriesHtml = '';
                    if (result.categories && result.categories.length > 0) {
                        categoriesHtml = '<p class="farmer-categories">' + result.categories.join(', ') + '</p>';
                    }
                    
                    // S'assurer que l'image est bien définie
                    let imageUrl = result.image || '';
                    // Si l'image est un chemin relatif, ajouter la base URL
                    if (imageUrl && !imageUrl.startsWith('http') && !imageUrl.startsWith('/')) {
                        imageUrl = root + imageUrl;
                    }
                    // Si aucune image n'est définie, utiliser une image par défaut
                    if (!imageUrl) {
                        imageUrl = root + 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                    }
                    
                    // Formater la distance (utiliser 0 comme valeur par défaut)
                    const distance = result.distance || 0;
                    const formattedDistance = Number.isFinite(distance) ? distance.toFixed(1) : "-";
                    
                    // Créer l'élément HTML
                    listItem.innerHTML = `
                        <a href="${profileUrl}" class="farmer-item" 
                           data-id="${submissionId}"
                           data-name="${result.name || ''}"
                           data-owner="${result.owner || ''}"
                           data-coordinates="${result.coordinates || ''}">
                            <div class="farmer-content">
                                <img src="${imageUrl}" alt="${result.name || ''}" class="farmer-image" onerror="this.src='${root}media/mod_agriculteur_liste/images/default-farmer.jpg'">
                                <div class="farmer-info">
                                    <h3 class="farmer-name">${result.name || ''}</h3>
                                    <p class="farmer-owner">${result.owner || ''}</p>
                                    ${categoriesHtml}
                                </div>
                                <div class="farmer-distance" id="distance-search-${submissionId}">
                                    <p class="distance-value">${formattedDistance}</p>
                                    <p class="distance-unit">km</p>
                                </div>
                            </div>
                        </a>
                    `;
                    
                    searchResultsList.appendChild(listItem);
                }
            });
            
            // Mettre à jour la pagination des résultats de recherche
            updateSearchPagination();
        }
        
        // Masquer les autres vues et afficher les résultats
        listView.classList.remove('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'block';
        
        // Désactiver tous les boutons de vue
        if (listViewButton) listViewButton.classList.remove('active');
        if (gridViewButton) gridViewButton.classList.remove('active');
        
        // Faire défiler vers le haut
        document.querySelector('.top-navigation-bar').scrollIntoView({ behavior: 'smooth' });
        
        // Si le bouton de géolocalisation existe, recalculer les distances pour les résultats
        if (geolocateButton && navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    const userLat = position.coords.latitude;
                    const userLng = position.coords.longitude;
                    
                    // Calculer les distances pour les résultats de recherche
                    searchResults.forEach(function(result) {
                        if (result.coordinates) {
                            const [farmLat, farmLng] = result.coordinates.split(',').map(parseFloat);
                            const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                            
                            // Mettre à jour l'affichage de la distance
                            const distanceElement = document.getElementById('distance-search-' + result.id);
                            if (distanceElement) {
                                const distanceValue = distanceElement.querySelector('.distance-value');
                                if (distanceValue) {
                                    distanceValue.textContent = distance.toFixed(1);
                                }
                            }
                        }
                    });
                },
                function(error) {
                    console.error('Erreur de géolocalisation:', error.message);
                }
            );
        }
    }
    
    // Fonction pour mettre à jour la pagination des résultats de recherche
    function updateSearchPagination() {
        const paginationContainer = document.querySelector('.pagination-container');
        if (!paginationContainer) return;
        
        // Calculer le nombre total de pages
        const totalSearchPages = Math.ceil(searchResults.length / searchItemsPerPage);
        
        // Si nous avons peu de résultats ou une seule page, masquer la pagination
        if (searchResults.length <= 5 || totalSearchPages <= 1) {
            paginationContainer.style.display = 'none';
            return;
        }
        
        // Sinon, afficher la pagination adaptée aux résultats de recherche
        paginationContainer.style.display = 'flex';
        
        // Mettre à jour l'affichage du nombre de pages
        const pageInfo = paginationContainer.querySelector('.page-info');
        if (pageInfo) {
            pageInfo.textContent = `Page ${searchCurrentPage} sur ${totalSearchPages}`;
        }
        
        // Mettre à jour les liens de pagination
        const pagination = paginationContainer.querySelector('.pagination');
        if (pagination) {
            // Vider la pagination actuelle
            pagination.innerHTML = '';
            
            // Bouton "Première page" si nécessaire
            if (searchCurrentPage > 1) {
                const firstPageButton = document.createElement('span');
                firstPageButton.className = 'page-item first';
                firstPageButton.innerHTML = '<i class="fas fa-angle-double-left"></i>';
                firstPageButton.style.cursor = 'pointer';
                firstPageButton.addEventListener('click', function() {
                    searchCurrentPage = 1;
                    displaySearchResults();
                });
                pagination.appendChild(firstPageButton);
                
                // Bouton "Page précédente"
                const prevPageButton = document.createElement('span');
                prevPageButton.className = 'page-item prev';
                prevPageButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
                prevPageButton.style.cursor = 'pointer';
                prevPageButton.addEventListener('click', function() {
                    searchCurrentPage--;
                    displaySearchResults();
                });
                pagination.appendChild(prevPageButton);
            }
            
            // Calcul des pages à afficher
            const maxPagesToShow = 5;
            let startPage = Math.max(1, searchCurrentPage - Math.floor(maxPagesToShow / 2));
            let endPage = Math.min(totalSearchPages, startPage + maxPagesToShow - 1);
            
            // Ajuster startPage si nécessaire
            startPage = Math.max(1, endPage - maxPagesToShow + 1);
            
            // Afficher la première page si nécessaire
            if (startPage > 1) {
                const firstPageLink = document.createElement('span');
                firstPageLink.className = 'page-item';
                firstPageLink.textContent = '1';
                firstPageLink.style.cursor = 'pointer';
                firstPageLink.addEventListener('click', function() {
                    searchCurrentPage = 1;
                    displaySearchResults();
                });
                pagination.appendChild(firstPageLink);
                
                // Ajouter des points de suspension si nécessaire
                if (startPage > 2) {
                    const ellipsis = document.createElement('span');
                    ellipsis.className = 'page-item ellipsis';
                    ellipsis.textContent = '...';
                    pagination.appendChild(ellipsis);
                }
            }
            
            // Afficher les numéros de page
            for (let i = startPage; i <= endPage; i++) {
                const pageLink = document.createElement('span');
                pageLink.className = i === searchCurrentPage ? 'page-item active' : 'page-item';
                pageLink.textContent = i;
                
                if (i !== searchCurrentPage) {
                    pageLink.style.cursor = 'pointer';
                    pageLink.addEventListener('click', function() {
                        searchCurrentPage = i;
                        displaySearchResults();
                    });
                }
                
                pagination.appendChild(pageLink);
            }
            
            // Afficher la dernière page si nécessaire
            if (endPage < totalSearchPages) {
                // Ajouter des points de suspension si nécessaire
                if (endPage < totalSearchPages - 1) {
                    const ellipsis = document.createElement('span');
                    ellipsis.className = 'page-item ellipsis';
                    ellipsis.textContent = '...';
                    pagination.appendChild(ellipsis);
                }
                
                const lastPageLink = document.createElement('span');
                lastPageLink.className = 'page-item';
                lastPageLink.textContent = totalSearchPages;
                lastPageLink.style.cursor = 'pointer';
                lastPageLink.addEventListener('click', function() {
                    searchCurrentPage = totalSearchPages;
                    displaySearchResults();
                });
                pagination.appendChild(lastPageLink);
            }
            
            // Bouton "Page suivante" si nécessaire
            if (searchCurrentPage < totalSearchPages) {
                const nextPageButton = document.createElement('span');
                nextPageButton.className = 'page-item next';
                nextPageButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
                nextPageButton.style.cursor = 'pointer';
                nextPageButton.addEventListener('click', function() {
                    searchCurrentPage++;
                    displaySearchResults();
                });
                pagination.appendChild(nextPageButton);
            }
        }
    }
    
    // Fonction pour masquer la pagination en mode recherche
    function hideSearchPagination() {
        const paginationContainer = document.querySelector('.pagination-container');
        if (paginationContainer) {
            paginationContainer.style.display = 'none';
        }
    }
    
    // Vérifier si on vient d'une URL avec des paramètres de recherche et de page
    function checkSearchParamsFromReturnURL() {
        const urlParams = new URLSearchParams(window.location.search);
        const searchQuery = urlParams.get('search_query');
        const searchPage = urlParams.get('search_page');
        
        if (searchQuery) {
            currentSearchQuery = decodeURIComponent(searchQuery);
            
            if (searchPage) {
                searchCurrentPage = parseInt(searchPage) || 1;
            }
            
            // Effectuer la recherche
            if (searchIndex) {
                performSearch(currentSearchQuery);
            } else {
                // Charger l'index et effectuer la recherche ensuite
                loadSearchIndex();
            }
            
            return true;
        }
        
        return false;
    }
    
    // Précharger l'index de recherche après le chargement initial de la page
    // pour que la recherche soit réactive dès la première utilisation
    setTimeout(function() {
        if (!searchIndex) {
            loadSearchIndex();
        }
        
        // Vérifier si on revient d'une page agriculteur avec des paramètres de recherche
        checkSearchParamsFromReturnURL();
    }, 500); // Délai court pour un chargement rapide
    
    // ====================
    // = GÉOLOCALISATION
    // ====================
    
    // Ajouter un gestionnaire d'événement au bouton de géolocalisation
    if (geolocateButton) {
        geolocateButton.addEventListener('click', getUserLocation);
    }
    
    // Fonction pour obtenir la position de l'utilisateur
    function getUserLocation() {
        if (!navigator.geolocation) {
            alert("La géolocalisation n'est pas supportée par votre navigateur.");
            return;
        }
        
        // Désactiver le bouton pendant la géolocalisation
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Localisation en cours...';
            geolocateButton.disabled = true;
        }
        
        navigator.geolocation.getCurrentPosition(
            // Succès
            function(position) {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                console.log('Position utilisateur obtenue:', userLat, userLng);
                
                // Afficher un message d'info
                if (distancesInfo) {
                    distancesInfo.style.display = 'block';
                }
                
                // Calculer les distances selon le mode actif
                if (isSearchMode) {
                    calculateSearchDistances(userLat, userLng);
                } else {
                    calculateDistances(userLat, userLng);
                }
                
                // Si la carte est initialisée, y ajouter la position de l'utilisateur
                if (map) {
                    const userPosition = new google.maps.LatLng(userLat, userLng);
                    
                    // Ajouter un marqueur pour l'utilisateur
                    const userMarker = new google.maps.Marker({
                        position: userPosition,
                        map: map,
                        title: 'Votre position',
                        icon: {
                            url: 'https://maps.google.com/mapfiles/ms/icons/blue-dot.png'
                        }
                    });
                    
                    // Centrer la carte sur l'utilisateur
                    map.setCenter(userPosition);
                    map.setZoom(12);
                }
            },
            // Erreur
            function(error) {
                console.error('Erreur de géolocalisation:', error.message);
                
                let errorMessage = "Impossible d'obtenir votre position.";
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += " Vous avez refusé l'accès à votre position.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += " Les informations de position ne sont pas disponibles.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += " La demande de position a expiré.";
                        break;
                    case error.UNKNOWN_ERROR:
                        errorMessage += " Une erreur inconnue s'est produite.";
                        break;
                }
                
                alert(errorMessage);
                
                // Réinitialiser le bouton
                if (geolocateButton) {
                    geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
                    geolocateButton.disabled = false;
                }
            },
            // Options
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            }
        );
    }
    
    // Fonction pour calculer les distances des résultats de recherche
    function calculateSearchDistances(userLat, userLng) {
        if (!searchResults || searchResults.length === 0) return;
        
        // Calculer les distances pour tous les résultats de recherche
        searchResults.forEach(function(result) {
            if (result.coordinates) {
                const [farmLat, farmLng] = result.coordinates.split(',').map(parseFloat);
                const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                
                // Stocker la distance dans le résultat
                result.distance = distance;
                
                // Mettre à jour l'affichage pour les résultats visibles
                const distanceElement = document.getElementById('distance-search-' + result.id);
                if (distanceElement) {
                    const distanceValue = distanceElement.querySelector('.distance-value');
                    if (distanceValue) {
                        distanceValue.textContent = distance.toFixed(1);
                    }
                }
            }
        });
        
        // Trier les résultats par distance
        searchResults.sort((a, b) => {
            const distA = a.distance !== undefined ? a.distance : Infinity;
            const distB = b.distance !== undefined ? b.distance : Infinity;
            return distA - distB;
        });
        
        // Réafficher les résultats triés
        displaySearchResults();
        
        // Réinitialiser le bouton
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            geolocateButton.disabled = false;
        }
        
        // Afficher un message de succès
        showSuccessMessage();
    }
    
    // Fonction pour calculer les distances vers tous les agriculteurs en vue normale
    function calculateDistances(userLat, userLng) {
        // Obtenir tous les éléments d'agriculteurs (liste et grille)
        const farmerItems = document.querySelectorAll('.farmer-item');
        const gridItems = document.querySelectorAll('.grid-item');
        
        // Calculer les distances à vol d'oiseau pour chaque agriculteur
        let distances = [];
        
        farmerItems.forEach(function(item) {
            const coordinates = item.getAttribute('data-coordinates');
            const id = item.getAttribute('data-id');
            
            if (coordinates) {
                const [farmLat, farmLng] = coordinates.split(',').map(parseFloat);
                const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                
                distances.push({
                    id: id,
                    distance: distance,
                    element: item
                });
                
                // Mettre à jour l'affichage dans la vue liste
                updateDistanceDisplay('distance-' + id, distance);
                
                // Mettre à jour l'affichage dans la vue grille
                updateDistanceDisplay('grid-distance-' + id, distance);
                
                // Stocker la distance dans l'élément pour le tri
                item.setAttribute('data-distance', distance);
                
                // Faire la même chose pour l'élément correspondant dans la vue grille
                const gridItem = document.querySelector(`.grid-item[data-id="${id}"]`);
                if (gridItem) {
                    gridItem.setAttribute('data-distance', distance);
                }
            }
        });
        
        // Trier les agriculteurs par distance
        distances.sort((a, b) => a.distance - b.distance);
        
        // Réorganiser les éléments dans le DOM (vue liste)
        const listContainer = document.getElementById('farmer-list-' + moduleId);
        if (listContainer) {
            const sortedItems = distances.map(item => item.element.parentNode);
            sortedItems.forEach(li => {
                if (li) listContainer.appendChild(li);
            });
        }
        
        // Réorganiser les éléments dans le DOM (vue grille)
        const gridContainer = document.getElementById('farmer-grid-' + moduleId);
        if (gridContainer) {
            const gridItems = Array.from(document.querySelectorAll('.grid-item')).sort((a, b) => {
                return parseFloat(a.getAttribute('data-distance')) - parseFloat(b.getAttribute('data-distance'));
            });
            gridItems.forEach(item => {
                gridContainer.appendChild(item);
            });
        }
        
        // Réinitialiser le bouton
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            geolocateButton.disabled = false;
        }
        
        // Afficher un message de succès
        showSuccessMessage();
    }
    
    // ====================
    // = GOOGLE MAPS
    // ====================
    
    // Chargement de l'API Google Maps et du script de clustering
    function loadGoogleMapsWithClusterer() {
        // Vérifier si Google Maps est déjà chargé
        if (typeof google === 'undefined' || typeof google.maps === 'undefined') {
            // Charger d'abord l'API Google Maps
            const gMapScript = document.createElement('script');
            gMapScript.src = 'https://maps.googleapis.com/maps/api/js?key=AIzaSyAjsOoV0528RtrLFvylBrsypk0Z3qik5RI&callback=initGoogleMapAfterLoad';
            gMapScript.async = true;
            gMapScript.defer = true;
            
            // Puis charger MarkerClustererPlus
            const clusterScript = document.createElement('script');
            clusterScript.src = 'https://unpkg.com/@googlemaps/markerclustererplus/dist/index.min.js';
            
            // Définir une fonction de rappel globale pour initialiser la carte après le chargement
            window.initGoogleMapAfterLoad = function() {
                console.log('Google Maps API chargée');
                // Attendre que le script de clustering soit également chargé
                if (typeof MarkerClusterer === 'undefined') {
                    setTimeout(initGoogleMapAfterLoad, 100); // Vérifier toutes les 100ms
                } else {
                    initMap();
                }
            };
            
            // Ajouter les scripts dans le bon ordre
            document.head.appendChild(clusterScript);
            document.head.appendChild(gMapScript);
        } else {
            // Google Maps est déjà chargé
            if (typeof MarkerClusterer === 'undefined') {
                // Charger uniquement le script de clustering
                const clusterScript = document.createElement('script');
                clusterScript.src = 'https://unpkg.com/@googlemaps/markerclustererplus/dist/index.min.js';
                clusterScript.onload = function() {
                    initMap();
                };
                document.head.appendChild(clusterScript);
            } else {
                // Tout est déjà chargé
                initMap();
            }
        }
    }
    
    // Initialisation de la carte Google Maps
    function initMap() {
        // Coordonnées centrales de La Réunion
        const reunionCenter = { lat: -21.115141, lng: 55.536384 };
        
        // Créer la carte avec les contrôles de zoom activés et bien visibles
        map = new google.maps.Map(document.getElementById('google-map'), {
            center: reunionCenter,
            zoom: 10,
            mapTypeId: google.maps.MapTypeId.ROADMAP,
            mapTypeControl: true,
            mapTypeControlOptions: {
                style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
                position: google.maps.ControlPosition.TOP_RIGHT
            },
            zoomControl: true,
            zoomControlOptions: {
                position: google.maps.ControlPosition.RIGHT_CENTER
            },
            streetViewControl: true,
            streetViewControlOptions: {
                position: google.maps.ControlPosition.RIGHT_BOTTOM
            },
            fullscreenControl: true
        });
        
        // Ajouter des marqueurs pour chaque agriculteur
        markers = [];
        const farmerItems = document.querySelectorAll('.farmer-item, .grid-item');
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            const coordinates = item.getAttribute('data-coordinates');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const profileUrl = item.href;
            
            if (coordinates) {
                const [lat, lng] = coordinates.split(',').map(parseFloat);
                
                // Créer un marqueur
                const marker = new google.maps.Marker({
                    position: { lat: lat, lng: lng },
                    map: map,
                    title: name,
                    icon: {
                        url: 'https://maps.google.com/mapfiles/ms/icons/green-dot.png'
                    }
                });
                
                // Créer un élément de contenu DOM au lieu d'une chaîne HTML
                // Cette approche contourne les restrictions CORS
                const infowindowContent = document.createElement('div');
                infowindowContent.className = 'custom-infowindow';
                
                const contentDiv = document.createElement('div');
                contentDiv.className = 'infowindow-content';
                
                const titleElement = document.createElement('h3');
                titleElement.className = 'infowindow-title';
                titleElement.textContent = name;
                
                const subtitleElement = document.createElement('p');
                subtitleElement.className = 'infowindow-subtitle';
                subtitleElement.textContent = owner;
                
                const linkElement = document.createElement('a');
                linkElement.className = 'infowindow-link';
                linkElement.href = profileUrl;
                linkElement.textContent = 'Voir le profil';
                linkElement.style.color = 'white';
                linkElement.style.backgroundColor = '#3c8f44';
                linkElement.style.padding = '5px 10px';
                linkElement.style.borderRadius = '3px';
                linkElement.style.textDecoration = 'none';
                linkElement.style.display = 'inline-block';
                linkElement.style.marginTop = '5px';
                
                // Si vous voulez tenter d'ajouter une image
                const imgContainer = document.createElement('div');
                imgContainer.className = 'infowindow-image-container';
                imgContainer.style.position = 'relative';
                imgContainer.style.width = '100%';
                imgContainer.style.paddingTop = '56.25%';
                imgContainer.style.overflow = 'hidden';
                
                const imgElement = document.createElement('img');
                imgElement.className = 'infowindow-image';
                imgElement.alt = name;
                imgElement.style.position = 'absolute';
                imgElement.style.top = '0';
                imgElement.style.left = '0';
                imgElement.style.width = '100%';
                imgElement.style.height = '100%';
                imgElement.style.objectFit = 'cover';
                
                // Trouver l'image dans l'item
                const existingImg = item.querySelector('img');
                if (existingImg && existingImg.src) {
                    imgElement.src = existingImg.src;
                }
                
                // Assembler les éléments
                imgContainer.appendChild(imgElement);
                contentDiv.appendChild(titleElement);
                contentDiv.appendChild(subtitleElement);
                contentDiv.appendChild(linkElement);
                infowindowContent.appendChild(imgContainer);
                infowindowContent.appendChild(contentDiv);
                
                // Créer l'infowindow avec le contenu DOM
                const infowindow = new google.maps.InfoWindow({
                    content: infowindowContent,
                    maxWidth: 300
                });
                
                // Ouvrir l'infowindow au clic
                marker.addListener('click', function() {
                    // Fermer tous les autres infowindows
                    markers.forEach(function(m) {
                        if (m.infowindow) {
                            m.infowindow.close();
                        }
                    });
                    
                    // Ouvrir l'infowindow de ce marqueur
                    infowindow.open(map, marker);
                });
                
                // Stocker le marqueur et son infowindow
                marker.infowindow = infowindow;
                markers.push(marker);
            }
        });
        
        // Initialiser le MarkerClusterer avec des options personnalisées
        if (markers.length > 0) {
            // Créer le marker clusterer
            markerCluster = new MarkerClusterer(map, markers, {
                imagePath: 'https://developers.google.com/maps/documentation/javascript/examples/markerclusterer/m',
                gridSize: 50,
                maxZoom: 15
            });
        }
    }
    
    // Charger Google Maps et le script de clustering lorsque le bouton de carte est cliqué
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            if (!map) {
                loadGoogleMapsWithClusterer();
            }
        });
    }
    
    // Mettre à jour les liens de pagination au chargement de la page
    updatePaginationLinks(activeView);
    
    // Vérifier si on revient d'une page avec des paramètres de recherche
    checkSearchParamsFromReturnURL();
});