<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_liste
 *
 * @copyright   Copyright (C) 2025 Votre Nom. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\HTML\HTMLHelper;

// Inclure le fichier d'aide du module de profil
require_once JPATH_SITE . '/modules/mod_agriculteur_profil/helper.php';

// Accès à la base de données
$db = Factory::getDbo();

// Récupérer les agriculteurs - on prend le formId des paramètres
$formId = $params->get('form_id', 1);

// Récupérer l'application
$app = Factory::getApplication();

// Récupérer la catégorie à filtrer depuis l'URL
$filterCategory = $app->input->getString('category', '');

// Liste des catégories valides
$validCategories = [
    'Agriculteur',
    'Commerce bio',
    'Transformateur',
    'Restaurateur',
    'Tables et gîte',
    'Autres'
];

// Requête pour récupérer tous les IDs
$query = $db->getQuery(true)
    ->select('s.SubmissionId')
    ->from($db->quoteName('#__rsform_submissions', 's'))
    ->where($db->quoteName('s.FormId') . ' = ' . $db->quote($formId))
    ->order($db->quoteName('s.DateSubmitted') . ' DESC');

$db->setQuery($query);
$submissionIds = $db->loadColumn();

// Initialiser le helper
$helper = new ModAgriculteurProfilHelper();

// Tableau pour stocker les agriculteurs et leurs distances
$agriculteurs = [];

// Position de l'utilisateur (à récupérer via HTML5 Geolocation API)
// Par défaut, coordonnées du centre de La Réunion
$userLat = -21.115141;
$userLng = 55.536384;

// Parcourir les soumissions et récupérer les données
foreach ($submissionIds as $submissionId) {
    $profileData = $helper->getProfileDataBySubmissionId($submissionId);
    
    if ($profileData) {
        // Vérifier si un filtre de catégorie est demandé
        if (!empty($filterCategory)) {
            // Récupérer les catégories de cet agriculteur
            $categoryRaw = isset($profileData['categories_raw']) ? $profileData['categories_raw'] : '';
            
            // Si la catégorie n'est pas présente dans la chaîne, passer au suivant
            // Note: nous vérifions si la catégorie est présente dans la chaîne, pas uniquement exactement égale
            if (strpos($categoryRaw, $filterCategory) === false) {
                continue;
            }
            
            // Ignorer "Petit Commerçant" qui est une ancienne entrée de test
            if (strpos($categoryRaw, 'Petit Commerçant') !== false) {
                continue;
            }
        }
        
        // Calculer la distance (à vol d'oiseau initialement)
        $distance = 0;
        
        if (!empty($profileData['geolocalisation'])) {
            list($agriLat, $agriLng) = explode(',', $profileData['geolocalisation']);
            
            // Distance à vol d'oiseau (formule Haversine)
            $distance = calculateDistance($userLat, $userLng, (float)$agriLat, (float)$agriLng);
        }
        
        // Récupérer l'URL de l'image de couverture
        $coverImageUrl = !empty($profileData['photo_couverture_petit_format_url']) 
            ? $profileData['photo_couverture_petit_format_url'] 
            : $helper->getUploadPath($profileData['photo_couverture_petit_format']);
            
        // Si pas d'image de couverture, utiliser une image par défaut
        if (empty($coverImageUrl)) {
            $coverImageUrl = Uri::root() . 'media/mod_agriculteur_liste/images/default-cover.jpg';
        }
        
        // Ajouter les données à notre tableau
        $agriculteurs[] = [
            'id' => $submissionId,
            'societe' => $profileData['societe'],
            'nom' => $profileData['nom'],
            'prenom' => $profileData['prenom'],
            'distance' => $distance,
            'image' => $coverImageUrl,
            'coordinates' => $profileData['geolocalisation']
        ];
    }
}

// Trier les agriculteurs par distance
usort($agriculteurs, function($a, $b) {
    return $a['distance'] <=> $b['distance'];
});

// Fonction pour calculer la distance à vol d'oiseau
function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371; // Rayon de la Terre en km
    
    $latFrom = deg2rad($lat1);
    $lonFrom = deg2rad($lon1);
    $latTo = deg2rad($lat2);
    $lonTo = deg2rad($lon2);
    
    $latDelta = $latTo - $latFrom;
    $lonDelta = $lonTo - $lonFrom;
    
    $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) + 
        cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
        
    return $angle * $earthRadius;
}

// Charger les styles et scripts nécessaires
$document = Factory::getDocument();

// Chargement des styles CSS
if ($params->get('load_styles', 1)) {
    $cssFile = 'media/mod_agriculteur_liste/css/style.css';
    if (file_exists(JPATH_SITE . '/' . $cssFile)) {
        HTMLHelper::_('stylesheet', $cssFile, ['version' => 'auto']);
    } else {
        // Fallback si le fichier CSS n'existe pas encore
        $document->addStyleDeclaration('
            .farmer-list {
                padding: 0;
                margin: 0;
                list-style: none;
            }
            
            .farmer-item {
                display: block;
                background-color: white;
                margin: 10px 0;
                border-radius: 0;
                overflow: hidden;
                box-shadow: 0 2px 5px rgba(0,0,0,0.1);
                text-decoration: none;
                color: inherit;
            }
            
            .farmer-content {
                display: flex;
                align-items: center;
                height: 100px;
            }
            
            .farmer-image {
                width: 100px;
                height: 100px;
                object-fit: cover;
                object-position: center;
                flex-shrink: 0;
            }
            
            .farmer-info {
                flex-grow: 1;
                padding: 10px;
                overflow: hidden;
            }
            
            .farmer-name {
                font-weight: bold;
                color: #3c8f44;
                margin: 0 0 5px 0;
                font-size: 1.1rem;
                white-space: nowrap;
                overflow: hidden;
                text-overflow: ellipsis;
            }
            
            .farmer-owner {
                margin: 0;
                font-size: 0.9rem;
                color: #666;
                white-space: nowrap;
                overflow: hidden;
                text-overflow: ellipsis;
            }
            
            .farmer-distance {
                padding: 10px 15px;
                text-align: center;
                background-color: #e9f5dc;
                min-width: 80px;
                display: flex;
                flex-direction: column;
                justify-content: center;
                flex-shrink: 0;
            }
            
            .distance-value {
                font-weight: bold;
                color: #333;
                font-size: 1.2rem;
                margin: 0;
            }
            
            .distance-unit {
                font-size: 0.8rem;
                color: #666;
                margin: 0;
            }
            
            .geolocation-button {
                display: block;
                margin: 10px auto;
                padding: 8px 15px;
                background-color: #3c8f44;
                color: white;
                border: none;
                border-radius: 5px;
                cursor: pointer;
                font-size: 0.9rem;
            }
        ');
    }
}

// Inclure le template d'affichage
require ModuleHelper::getLayoutPath('mod_agriculteur_liste', $params->get('layout', 'default'));