<?php
/**
 * API pour calculer la distance routière entre 2 points avec OpenRouteService
 * Fallback vers calcul à vol d'oiseau si ORS échoue
 * 
 * Usage: get-distance.php?start=lon,lat&end=lon,lat&fallback=1
 */

// Définir que la réponse sera en JSON
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Gérer les requêtes OPTIONS (preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Vérifier les paramètres requis
if (!isset($_GET['start']) || !isset($_GET['end'])) {
    http_response_code(400);
    echo json_encode([
        'error' => 'Paramètres manquants: start et end sont requis',
        'usage' => 'get-distance.php?start=lon,lat&end=lon,lat&fallback=1'
    ]);
    exit;
}

// Récupérer les coordonnées
$start = trim($_GET['start']);
$end = trim($_GET['end']);
$enableFallback = isset($_GET['fallback']) && $_GET['fallback'] == '1';

// Valider le format des coordonnées
if (!preg_match('/^-?\d+\.?\d*,-?\d+\.?\d*$/', $start) || !preg_match('/^-?\d+\.?\d*,-?\d+\.?\d*$/', $end)) {
    http_response_code(400);
    echo json_encode([
        'error' => 'Format de coordonnées invalide',
        'expected' => 'longitude,latitude (ex: 55.4504,-20.8823)'
    ]);
    exit;
}

// Clé API OpenRouteService - À REMPLACER par votre vraie clé !
$apiKey = '5b3ce3597851110001cf62480c7ae4d074084cd6aefd8f3652182e26';

// Fonction pour calculer la distance à vol d'oiseau (fallback)
function calculateHaversineDistance($start, $end) {
    list($lon1, $lat1) = explode(',', $start);
    list($lon2, $lat2) = explode(',', $end);
    
    $earthRadius = 6371; // Rayon de la Terre en km
    
    $latFrom = deg2rad((float)$lat1);
    $lonFrom = deg2rad((float)$lon1);
    $latTo = deg2rad((float)$lat2);
    $lonTo = deg2rad((float)$lon2);
    
    $latDelta = $latTo - $latFrom;
    $lonDelta = $lonTo - $lonFrom;
    
    $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) + 
        cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
        
    return $angle * $earthRadius;
}

// Essayer d'abord OpenRouteService
try {
    // Construction de l'URL pour l'API ORS (format correct)
    $baseUrl = "https://api.openrouteservice.org/v2/directions/driving-car";
    
    // Construire le body de la requête POST (recommandé par ORS)
    $requestData = [
        'coordinates' => [
            explode(',', $start), // [lon, lat]
            explode(',', $end)    // [lon, lat]
        ],
        'format' => 'json'
    ];
    
    // Initialiser cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $baseUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'Authorization: ' . $apiKey,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);   // Timeout de connexion court
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);         // Timeout total
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Agriculteur-Module/1.0');
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    
    curl_close($ch);
    
    // Vérifier les erreurs cURL
    if ($curlError) {
        throw new Exception("Erreur cURL: " . $curlError);
    }
    
    // Vérifier le code de réponse HTTP
    if ($httpCode !== 200) {
        throw new Exception("Erreur HTTP: " . $httpCode . " - " . $response);
    }
    
    // Décoder la réponse JSON
    $data = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Erreur JSON: " . json_last_error_msg());
    }
    
    // Vérifier la structure de la réponse ORS
    if (!isset($data['routes']) || !isset($data['routes'][0]) || !isset($data['routes'][0]['summary'])) {
        throw new Exception("Structure de réponse ORS invalide");
    }
    
    $route = $data['routes'][0];
    $distance = $route['summary']['distance']; // Distance en mètres
    $duration = $route['summary']['duration']; // Durée en secondes
    
    // Calculer la distance à vol d'oiseau pour comparaison
    $straightDistance = calculateHaversineDistance($start, $end);
    
    // Retourner la réponse formatée
    echo json_encode([
        'success' => true,
        'source' => 'openrouteservice',
        'distance' => [
            'road_km' => round($distance / 1000, 2),
            'road_meters' => $distance,
            'straight_km' => round($straightDistance, 2)
        ],
        'duration' => [
            'seconds' => $duration,
            'minutes' => round($duration / 60, 1),
            'formatted' => gmdate('H:i', $duration)
        ],
        'detour_factor' => round(($distance / 1000) / $straightDistance, 2),
        'coordinates' => [
            'start' => $start,
            'end' => $end
        ]
    ]);
    
} catch (Exception $e) {
    // Si OpenRouteService échoue ET que le fallback est activé
    if ($enableFallback) {
        $straightDistance = calculateHaversineDistance($start, $end);
        
        echo json_encode([
            'success' => true,
            'source' => 'haversine_fallback',
            'warning' => 'OpenRouteService indisponible, calcul à vol d\'oiseau utilisé',
            'ors_error' => $e->getMessage(),
            'distance' => [
                'road_km' => $straightDistance, // Utilise vol d'oiseau comme approximation
                'straight_km' => $straightDistance
            ],
            'duration' => [
                'estimated_minutes' => round($straightDistance * 1.5, 1) // Estimation basique
            ],
            'detour_factor' => 1.0,
            'coordinates' => [
                'start' => $start,
                'end' => $end
            ]
        ]);
    } else {
        // Retourner l'erreur
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'error' => 'OpenRouteService indisponible',
            'details' => $e->getMessage(),
            'suggestion' => 'Ajoutez &fallback=1 pour utiliser le calcul à vol d\'oiseau'
        ]);
    }
}
?>