<?php
/**
 * Script de validation OpenRouteService pour le module Agriculteur Liste
 * 
 * Usage: Placez ce fichier dans /modules/mod_agriculteur_liste/test/
 * Accédez à : votre-site.com/modules/mod_agriculteur_liste/test/validate-ors.php
 */

// Configuration
$apiKey = '5b3ce3597851110001cf62480c7ae4d074084cd6aefd8f3652182e26'; // ⬅️ Remplacez par votre vraie clé
$testRoutes = [
    [
        'name' => 'Saint-Denis → Saint-Pierre',
        'start' => [55.4504, -20.8823], // [lon, lat]
        'end' => [55.4781, -21.3393],
        'expected_min' => 75, // Distance minimale attendue en km
        'expected_max' => 85  // Distance maximale attendue en km
    ],
    [
        'name' => 'Saint-Denis → Cilaos',
        'start' => [55.4504, -20.8823],
        'end' => [55.4732, -21.1322],
        'expected_min' => 100,
        'expected_max' => 110
    ],
    [
        'name' => 'Saint-Paul → Saint-Benoît',
        'start' => [55.2697, -21.0099],
        'end' => [55.7135, -21.0344],
        'expected_min' => 65,
        'expected_max' => 85
    ]
];

?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>🧪 Validation OpenRouteService - Module Agriculteur</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            max-width: 1000px;
            margin: 0 auto;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            text-align: center;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }
        .test-card {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 20px;
            margin: 15px 0;
            position: relative;
        }
        .test-card.success {
            border-left: 4px solid #28a745;
            background: #d4edda;
        }
        .test-card.error {
            border-left: 4px solid #dc3545;
            background: #f8d7da;
        }
        .test-card.warning {
            border-left: 4px solid #ffc107;
            background: #fff3cd;
        }
        .test-title {
            font-size: 1.1em;
            font-weight: bold;
            margin-bottom: 10px;
        }
        .test-details {
            font-size: 0.9em;
            color: #666;
            margin: 5px 0;
        }
        .metrics {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 10px;
            margin: 15px 0;
        }
        .metric {
            background: white;
            padding: 10px;
            border-radius: 5px;
            text-align: center;
            border: 1px solid #ddd;
        }
        .metric-value {
            font-size: 1.3em;
            font-weight: bold;
            color: #2c3e50;
        }
        .metric-label {
            font-size: 0.8em;
            color: #666;
            margin-top: 5px;
        }
        .config-section {
            background: #e9ecef;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .alert {
            padding: 12px;
            border-radius: 5px;
            margin: 15px 0;
        }
        .alert.info {
            background: #d1ecf1;
            border: 1px solid #bee5eb;
            color: #0c5460;
        }
        .alert.success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        .alert.error {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        .status-indicator {
            position: absolute;
            top: 10px;
            right: 15px;
            font-size: 1.5em;
        }
        .loading {
            color: #ffc107;
        }
        pre {
            background: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            border: 1px solid #e9ecef;
            overflow-x: auto;
            font-size: 0.85em;
        }
        .test-button {
            background: #007bff;
            color: white;
            border: none;
            padding: 12px 25px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 16px;
            margin: 10px 5px;
        }
        .test-button:hover {
            background: #0056b3;
        }
        .test-button:disabled {
            background: #6c757d;
            cursor: not-allowed;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧪 Validation OpenRouteService</h1>
        
        <div class="config-section">
            <h3>🔧 Configuration</h3>
            <p><strong>Clé API :</strong> 
                <?php if ($apiKey === 'VOTRE_CLE_API_ICI'): ?>
                    <span style="color: #dc3545;">❌ Non configurée - Modifiez la variable $apiKey</span>
                <?php else: ?>
                    <span style="color: #28a745;">✅ Configurée (<?php echo substr($apiKey, 0, 20); ?>...)</span>
                <?php endif; ?>
            </p>
            <p><strong>Tests prévus :</strong> <?php echo count($testRoutes); ?> routes de La Réunion</p>
            <p><strong>But :</strong> Valider le fonctionnement d'OpenRouteService avec des routes réelles</p>
        </div>

        <?php if ($apiKey === 'VOTRE_CLE_API_ICI'): ?>
            <div class="alert error">
                <strong>⚠️ Configuration requise</strong><br>
                Veuillez modifier la variable <code>$apiKey</code> dans ce script avec votre vraie clé OpenRouteService.
            </div>
        <?php else: ?>
            
            <button id="run-tests" class="test-button" onclick="runAllTests()">🚀 Lancer tous les tests</button>
            <button class="test-button" onclick="testApiConnection()">🔌 Tester connexion API</button>
            <button class="test-button" onclick="location.reload()">🔄 Actualiser</button>
            
            <div id="test-results">
                <!-- Les résultats apparaîtront ici -->
            </div>
            
        <?php endif; ?>
    </div>

    <script>
        const API_KEY = '<?php echo $apiKey; ?>';
        const TEST_ROUTES = <?php echo json_encode($testRoutes); ?>;
        
        let testResults = [];
        
        // Fonction pour calculer la distance Haversine (référence)
        function calculateHaversineDistance(lat1, lon1, lat2, lon2) {
            const R = 6371;
            const dLat = (lat2 - lat1) * Math.PI / 180;
            const dLon = (lon2 - lon1) * Math.PI / 180;
            const a = Math.sin(dLat/2) * Math.sin(dLat/2) +
                    Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) *
                    Math.sin(dLon/2) * Math.sin(dLon/2);
            const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
            return R * c;
        }
        
        // Test de connexion à l'API
        async function testApiConnection() {
            const resultsDiv = document.getElementById('test-results');
            resultsDiv.innerHTML = `
                <div class="test-card">
                    <span class="status-indicator loading">⏳</span>
                    <div class="test-title">Test de connexion API OpenRouteService</div>
                    <div class="test-details">Vérification de la validité de la clé API...</div>
                </div>
            `;
            
            try {
                const response = await fetch('https://api.openrouteservice.org/v2/directions/driving-car', {
                    method: 'POST',
                    headers: {
                        'Accept': 'application/json',
                        'Authorization': API_KEY,
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        coordinates: [[55.4504, -20.8823], [55.4781, -21.3393]],
                        format: 'json'
                    })
                });
                
                if (response.ok) {
                    const data = await response.json();
                    resultsDiv.innerHTML = `
                        <div class="test-card success">
                            <span class="status-indicator">✅</span>
                            <div class="test-title">Connexion API réussie</div>
                            <div class="test-details">
                                Status: ${response.status}<br>
                                Serveur: ${response.headers.get('Server') || 'OpenRouteService'}<br>
                                Route calculée: ${(data.routes[0].summary.distance / 1000).toFixed(1)} km
                            </div>
                        </div>
                    `;
                } else {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
            } catch (error) {
                resultsDiv.innerHTML = `
                    <div class="test-card error">
                        <span class="status-indicator">❌</span>
                        <div class="test-title">Erreur de connexion API</div>
                        <div class="test-details">
                            Erreur: ${error.message}<br><br>
                            <strong>Causes possibles:</strong><br>
                            • Clé API invalide<br>
                            • Quota dépassé<br>
                            • Problème de connectivité<br>
                            • CORS (utilisez le script via serveur web)
                        </div>
                    </div>
                `;
            }
        }
        
        // Test d'une route spécifique
        async function testRoute(route, index) {
            const startTime = Date.now();
            
            try {
                const response = await fetch('https://api.openrouteservice.org/v2/directions/driving-car', {
                    method: 'POST',
                    headers: {
                        'Accept': 'application/json',
                        'Authorization': API_KEY,
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        coordinates: [route.start, route.end],
                        format: 'json'
                    })
                });
                
                const responseTime = Date.now() - startTime;
                
                if (!response.ok) {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
                
                const data = await response.json();
                const routeInfo = data.routes[0];
                const distance = routeInfo.summary.distance / 1000; // en km
                const duration = routeInfo.summary.duration / 60; // en minutes
                
                // Calculer la distance à vol d'oiseau pour comparaison
                const straightDistance = calculateHaversineDistance(
                    route.start[1], route.start[0], route.end[1], route.end[0]
                );
                const detourFactor = distance / straightDistance;
                
                // Vérifier si c'est dans les attentes
                const isValid = distance >= route.expected_min && distance <= route.expected_max;
                
                return {
                    success: true,
                    route: route,
                    distance: distance,
                    duration: duration,
                    straightDistance: straightDistance,
                    detourFactor: detourFactor,
                    responseTime: responseTime,
                    isValid: isValid
                };
                
            } catch (error) {
                return {
                    success: false,
                    route: route,
                    error: error.message,
                    responseTime: Date.now() - startTime
                };
            }
        }
        
        // Lancer tous les tests
        async function runAllTests() {
            const button = document.getElementById('run-tests');
            button.disabled = true;
            button.textContent = '⏳ Tests en cours...';
            
            const resultsDiv = document.getElementById('test-results');
            resultsDiv.innerHTML = '<div class="alert info">🧪 Exécution des tests en cours...</div>';
            
            testResults = [];
            let html = '';
            
            for (let i = 0; i < TEST_ROUTES.length; i++) {
                const route = TEST_ROUTES[i];
                
                // Afficher le test en cours
                html += `
                    <div class="test-card" id="test-${i}">
                        <span class="status-indicator loading">⏳</span>
                        <div class="test-title">${route.name}</div>
                        <div class="test-details">Test en cours...</div>
                    </div>
                `;
                resultsDiv.innerHTML = html;
                
                // Exécuter le test
                const result = await testRoute(route, i);
                testResults.push(result);
                
                // Mettre à jour l'affichage
                if (result.success) {
                    const cardClass = result.isValid ? 'success' : 'warning';
                    const statusIcon = result.isValid ? '✅' : '⚠️';
                    
                    document.getElementById(`test-${i}`).outerHTML = `
                        <div class="test-card ${cardClass}">
                            <span class="status-indicator">${statusIcon}</span>
                            <div class="test-title">${result.route.name}</div>
                            <div class="test-details">
                                Attendu: ${result.route.expected_min}-${result.route.expected_max} km | 
                                Obtenu: ${result.distance.toFixed(1)} km
                                ${result.isValid ? '' : ' (Hors limites attendues)'}
                            </div>
                            <div class="metrics">
                                <div class="metric">
                                    <div class="metric-value">${result.distance.toFixed(1)} km</div>
                                    <div class="metric-label">Distance route</div>
                                </div>
                                <div class="metric">
                                    <div class="metric-value">${result.straightDistance.toFixed(1)} km</div>
                                    <div class="metric-label">Vol d'oiseau</div>
                                </div>
                                <div class="metric">
                                    <div class="metric-value">${result.duration.toFixed(0)} min</div>
                                    <div class="metric-label">Durée estimée</div>
                                </div>
                                <div class="metric">
                                    <div class="metric-value">${result.detourFactor.toFixed(2)}x</div>
                                    <div class="metric-label">Facteur détour</div>
                                </div>
                                <div class="metric">
                                    <div class="metric-value">${result.responseTime} ms</div>
                                    <div class="metric-label">Temps réponse</div>
                                </div>
                            </div>
                        </div>
                    `;
                } else {
                    document.getElementById(`test-${i}`).outerHTML = `
                        <div class="test-card error">
                            <span class="status-indicator">❌</span>
                            <div class="test-title">${result.route.name}</div>
                            <div class="test-details">
                                Erreur: ${result.error}<br>
                                Temps: ${result.responseTime} ms
                            </div>
                        </div>
                    `;
                }
                
                // Pause entre les tests pour éviter la limitation de débit
                if (i < TEST_ROUTES.length - 1) {
                    await new Promise(resolve => setTimeout(resolve, 1000));
                }
            }
            
            // Afficher le résumé
            const successCount = testResults.filter(r => r.success).length;
            const validCount = testResults.filter(r => r.success && r.isValid).length;
            const avgResponseTime = testResults
                .filter(r => r.success)
                .reduce((sum, r) => sum + r.responseTime, 0) / successCount;
            
            const summaryClass = validCount === TEST_ROUTES.length ? 'success' : 
                                 successCount === TEST_ROUTES.length ? 'warning' : 'error';
            
            resultsDiv.innerHTML += `
                <div class="test-card ${summaryClass}">
                    <span class="status-indicator">${validCount === TEST_ROUTES.length ? '🎉' : successCount === TEST_ROUTES.length ? '⚠️' : '❌'}</span>
                    <div class="test-title">Résumé des tests</div>
                    <div class="metrics">
                        <div class="metric">
                            <div class="metric-value">${successCount}/${TEST_ROUTES.length}</div>
                            <div class="metric-label">Tests réussis</div>
                        </div>
                        <div class="metric">
                            <div class="metric-value">${validCount}/${TEST_ROUTES.length}</div>
                            <div class="metric-label">Résultats valides</div>
                        </div>
                        <div class="metric">
                            <div class="metric-value">${successCount > 0 ? avgResponseTime.toFixed(0) : 0} ms</div>
                            <div class="metric-label">Temps moyen</div>
                        </div>
                    </div>
                    <div class="test-details">
                        ${validCount === TEST_ROUTES.length ? 
                            '🎉 Tous les tests sont valides ! OpenRouteService fonctionne parfaitement pour La Réunion.' :
                            successCount === TEST_ROUTES.length ?
                            '⚠️ API fonctionnelle mais certaines distances sont hors des attentes. Vérifiez les routes.' :
                            '❌ Certains tests ont échoué. Vérifiez votre configuration et connectivité.'
                        }
                    </div>
                </div>
            `;
            
            button.disabled = false;
            button.textContent = '🚀 Lancer tous les tests';
        }
    </script>
</body>
</html>