/**
 * Script principal du module agriculteur_liste avec debug distances
 */

document.addEventListener('DOMContentLoaded', function() {
    // Variables et éléments du DOM
    const geolocateButton = document.getElementById('geolocate-button-' + moduleId);
    const distancesInfo = document.getElementById('distances-info-' + moduleId);
    
    // Références aux boutons de navigation
    const mapViewButton = document.getElementById('map-view-button');
    const gridViewButton = document.getElementById('grid-view-button');
    const searchButton = document.getElementById('search-button');
    
    // Références aux conteneurs de vues
    const mapContainer = document.getElementById('map-container');
    const gridContainer = document.getElementById('grid-container');
    
    // Références aux vues liste et grille
    const listView = document.getElementById('list-view');
    const gridView = document.getElementById('grid-view');
    
    // Références au popup de recherche et aux éléments associés
    const searchPopup = document.getElementById('search-popup');
    const searchInput = document.getElementById('search-input');
    const searchSubmit = document.getElementById('search-submit');
    const searchClose = document.getElementById('search-close');
    const searchResultsContainer = document.getElementById('search-results-container');
    const searchResultsList = document.getElementById('search-results-list');
    const searchCount = document.getElementById('search-count');
    
    // Variables pour le mode recherche
    let isSearchMode = false;
    let currentSearchQuery = '';
    let searchResults = [];
    let searchCurrentPage = 1;
    const searchItemsPerPage = 10;
    
    // Cache pour l'index de recherche
    let searchIndex = null;
    
    // Variables pour la carte Google Maps
    let map;
    let markers = [];
    let markerCluster;
    
    // Configuration pour OpenRouteService
    const orsConfig = window.agriculteurModule?.config || {
        useOpenRouteService: true,
        distanceApiUrl: root + 'modules/mod_agriculteur_liste/api/get-distance.php'
    };
    
    // ====================
    // = FONCTIONS UTILITAIRES
    // ====================
    
    function setViewMode(mode) {
        try {
            localStorage.setItem('agriculteur_view_mode', mode);
        } catch (e) {
            console.error('Impossible de stocker le mode de vue:', e);
        }
        updatePaginationLinks(mode);
    }
    
    function updatePaginationLinks(viewMode) {
        document.querySelectorAll('.pagination a').forEach(function(link) {
            const url = new URL(link.href);
            url.searchParams.set('view_mode', viewMode);
            
            if (isSearchMode && currentSearchQuery) {
                url.searchParams.set('search_query', encodeURIComponent(currentSearchQuery));
            }
            
            link.href = url.toString();
        });
    }
    
    function toRad(degrees) {
        return degrees * Math.PI / 180;
    }
    
    // Fonction pour calculer la distance à vol d'oiseau (Haversine) avec DEBUG
    function calculateHaversineDistance(lat1, lng1, lat2, lng2) {
        console.log(`🧮 DEBUG Haversine: [${lat1}, ${lng1}] → [${lat2}, ${lng2}]`);
        
        // Vérifier que toutes les valeurs sont des nombres valides
        if (isNaN(lat1) || isNaN(lng1) || isNaN(lat2) || isNaN(lng2)) {
            console.error('❌ Coordonnées invalides:', { lat1, lng1, lat2, lng2 });
            return 0;
        }
        
        const R = 6371; // Rayon de la Terre en km
        const dLat = toRad(lat2 - lat1);
        const dLng = toRad(lng2 - lng1);
        
        const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) *
                Math.sin(dLng / 2) * Math.sin(dLng / 2);
                
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        const distance = R * c;
        
        console.log(`📏 Distance calculée: ${distance.toFixed(2)} km`);
        return distance;
    }
    
    // Fonction pour calculer la distance routière avec OpenRouteService et DEBUG
    async function calculateRouteDistance(userLat, userLng, farmLat, farmLng) {
        console.log(`🗺️ DEBUG ORS: User[${userLat}, ${userLng}] → Farm[${farmLat}, ${farmLng}]`);
        
        try {
            const url = `${orsConfig.distanceApiUrl}?start=${userLng},${userLat}&end=${farmLng},${farmLat}&fallback=1`;
            console.log('🌐 URL appelée:', url);
            
            const startTime = Date.now();
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Accept': 'application/json'
                }
            });
            const responseTime = Date.now() - startTime;
            
            console.log(`⏱️ Temps réponse: ${responseTime}ms`);
            
            if (!response.ok) {
                throw new Error(`Erreur HTTP: ${response.status}`);
            }
            
            const data = await response.json();
            console.log('📊 Réponse complète:', data);
            
            if (data.success && data.distance && data.distance.road_km) {
                console.log(`✅ Distance ORS: ${data.distance.road_km} km (source: ${data.source})`);
                return {
                    distance: data.distance.road_km,
                    source: data.source,
                    duration: data.duration,
                    detourFactor: data.detour_factor
                };
            } else {
                throw new Error('Réponse ORS invalide: ' + JSON.stringify(data));
            }
            
        } catch (error) {
            console.warn('⚠️ Erreur OpenRouteService, fallback Haversine:', error.message);
            
            // Fallback vers calcul à vol d'oiseau
            const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
            return {
                distance: distance,
                source: 'haversine_fallback',
                duration: null,
                detourFactor: 1.0
            };
        }
    }
    
    function updateDistanceDisplay(elementId, distanceData) {
        const distanceElement = document.getElementById(elementId);
        if (!distanceElement) {
            console.warn(`⚠️ Élément distance non trouvé: ${elementId}`);
            return;
        }
        
        const distanceValue = distanceElement.querySelector('.distance-value') || 
                            distanceElement.querySelector('.grid-distance-value');
        
        if (distanceValue) {
            const formattedDistance = distanceData.distance.toFixed(1);
            console.log(`📊 Mise à jour ${elementId}: ${formattedDistance} km (${distanceData.source})`);
            distanceValue.textContent = formattedDistance;
        }
    }

    function showSuccessMessage(stats) {
        const successMessage = document.createElement('div');
        successMessage.className = 'geolocation-success';
        
        let message = '<i class="fas fa-check-circle"></i> Distances recalculées !';
        if (stats) {
            message += `<br><small>Routes: ${stats.ors} | Vol d'oiseau: ${stats.fallback}</small>`;
        }
        
        successMessage.innerHTML = message;
        document.body.appendChild(successMessage);
        
        setTimeout(function() {
            successMessage.remove();
        }, 4000);
    }
    
    // ====================
    // = GESTION DES VUES
    // ====================
    
    const listViewButton = document.createElement('button');
    listViewButton.innerHTML = '<i class="fas fa-list"></i>';
    listViewButton.className = 'nav-button list-view-button';
    listViewButton.style.marginRight = '0px';
    
    if (gridViewButton && gridViewButton.parentNode) {
        gridViewButton.parentNode.insertBefore(listViewButton, gridViewButton);
    }
    
    if (activeView === 'list') {
        listViewButton.classList.add('active');
    }
    
    function checkForSearchQueryInURL() {
        const urlParams = new URLSearchParams(window.location.search);
        const searchQuery = urlParams.get('search_query');
        
        if (searchQuery) {
            const decodedQuery = decodeURIComponent(searchQuery);
            currentSearchQuery = decodedQuery;
            
            if (searchInput) {
                searchInput.value = decodedQuery;
            }
            
            if (!searchIndex) {
                loadSearchIndex();
                setTimeout(() => {
                    performSearch(decodedQuery);
                }, 500);
            } else {
                performSearch(decodedQuery);
            }
            
            return true;
        }
        
        return false;
    }
    
    // Gestionnaires d'événements pour les boutons de navigation
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            isSearchMode = false;
            mapViewButton.classList.add('active');
            gridViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            mapContainer.style.display = 'block';
            gridContainer.style.display = 'none';
            
            if (!map) {
                loadGoogleMapsWithClusterer();
            } else {
                google.maps.event.trigger(map, 'resize');
            }
            
            setViewMode('map');
        });
    }
    
    if (gridViewButton) {
        gridViewButton.addEventListener('click', function() {
            isSearchMode = false;
            gridViewButton.classList.add('active');
            mapViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            gridContainer.style.display = 'block';
            mapContainer.style.display = 'none';
            gridView.classList.add('active-view');
            listView.classList.remove('active-view');
            searchResultsContainer.style.display = 'none';
            setViewMode('grid');
        });
    }
    
    listViewButton.addEventListener('click', function() {
        isSearchMode = false;
        
        if (mapContainer.style.display !== 'none') {
            mapViewButton.classList.remove('active');
            gridViewButton.classList.remove('active');
            mapContainer.style.display = 'none';
            gridContainer.style.display = 'block';
        }
        
        listView.classList.add('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'none';
        listViewButton.classList.add('active');
        gridViewButton.classList.remove('active');
        setViewMode('list');
    });
    
    // ====================
    // = RECHERCHE (code simplifié pour se concentrer sur les distances)
    // ====================
    
    if (searchButton) {
        searchButton.addEventListener('click', function() {
            searchPopup.style.display = 'flex';
            searchInput.focus();
            
            if (currentSearchQuery) {
                searchInput.value = currentSearchQuery;
            } else {
                searchInput.value = '';
            }
            
            if (!searchIndex) {
                loadSearchIndex();
            }
        });
    }
    
    if (searchClose) {
        searchClose.addEventListener('click', function() {
            searchPopup.style.display = 'none';
        });
    }
    
    searchPopup.addEventListener('click', function(event) {
        if (event.target === searchPopup) {
            searchPopup.style.display = 'none';
        }
    });
    
    function loadSearchIndex() {
        // Simplifié pour se concentrer sur les distances
        console.log('Chargement de l\'index de recherche simplifié...');
        const farmerItems = document.querySelectorAll('.farmer-item, .grid-item');
        searchIndex = [];
        
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const coordinates = item.getAttribute('data-coordinates');
            
            const searchableText = [name || '', owner || ''].join(' ').toLowerCase();
            
            let image = '';
            const imgElement = item.querySelector('img');
            if (imgElement && imgElement.src) {
                image = imgElement.src;
            }
            
            searchIndex.push({
                id: id,
                name: name || '',
                owner: owner || '',
                coordinates: coordinates || '',
                searchableText: searchableText,
                image: image,
                element: item
            });
        });
        
        console.log('Index de recherche chargé avec ' + searchIndex.length + ' entrées');
        checkForSearchQueryInURL();
    }
    
    // ====================
    // = GÉOLOCALISATION avec DEBUG
    // ====================
    
    if (geolocateButton) {
        geolocateButton.addEventListener('click', getUserLocation);
    }
    
    function getUserLocation() {
        console.log('🎯 Début géolocalisation...');
        
        if (!navigator.geolocation) {
            alert("La géolocalisation n'est pas supportée par votre navigateur.");
            return;
        }
        
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Localisation en cours...';
            geolocateButton.disabled = true;
        }
        
        navigator.geolocation.getCurrentPosition(
            // Succès
            function(position) {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                console.log('🎯 Position utilisateur obtenue:', { userLat, userLng });
                console.log('📍 Précision:', position.coords.accuracy, 'mètres');
                
                if (distancesInfo) {
                    distancesInfo.innerHTML = '<i class="fas fa-info-circle"></i> Calcul des distances en cours...';
                    distancesInfo.style.display = 'block';
                }
                
                // Calculer les distances selon le mode actif
                if (isSearchMode) {
                    calculateSearchDistances(userLat, userLng);
                } else {
                    calculateDistances(userLat, userLng);
                }
            },
            // Erreur
            function(error) {
                console.error('❌ Erreur de géolocalisation:', error);
                
                let errorMessage = "Impossible d'obtenir votre position.";
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += " Vous avez refusé l'accès à votre position.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += " Les informations de position ne sont pas disponibles.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += " La demande de position a expiré.";
                        break;
                    case error.UNKNOWN_ERROR:
                        errorMessage += " Une erreur inconnue s'est produite.";
                        break;
                }
                
                alert(errorMessage);
                
                if (geolocateButton) {
                    geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
                    geolocateButton.disabled = false;
                }
            },
            // Options
            {
                enableHighAccuracy: true,
                timeout: 10000,
                maximumAge: 0
            }
        );
    }
    
    // Fonction pour calculer les distances vers tous les agriculteurs avec DEBUG
    async function calculateDistances(userLat, userLng) {
        console.log('🧮 DÉBUT calcul distances pour tous les agriculteurs');
        console.log('👤 Position utilisateur:', { userLat, userLng });
        
        const farmerItems = document.querySelectorAll('.farmer-item');
        const gridItems = document.querySelectorAll('.grid-item');
        
        console.log(`📊 Nombre d'agriculteurs trouvés: ${farmerItems.length}`);
        
        let distances = [];
        let orsCount = 0;
        let fallbackCount = 0;
        
        // Calculer les distances pour chaque agriculteur
        const promises = Array.from(farmerItems).map(async (item, index) => {
            const coordinates = item.getAttribute('data-coordinates');
            const id = item.getAttribute('data-id');
            const name = item.getAttribute('data-name');
            
            console.log(`\n🌱 Agriculteur ${index + 1}/${farmerItems.length}: ${name} (ID: ${id})`);
            console.log(`📍 Coordonnées brutes: "${coordinates}"`);
            
            if (!coordinates || coordinates === '' || coordinates === 'null' || coordinates === 'undefined') {
                console.error(`❌ Coordonnées manquantes pour ${name}`);
                return;
            }
            
            // Parser les coordonnées
            const coordParts = coordinates.split(',');
            if (coordParts.length !== 2) {
                console.error(`❌ Format coordonnées invalide pour ${name}: "${coordinates}"`);
                return;
            }
            
            const farmLat = parseFloat(coordParts[0].trim());
            const farmLng = parseFloat(coordParts[1].trim());
            
            console.log(`📍 Coordonnées parsées: [${farmLat}, ${farmLng}]`);
            
            if (isNaN(farmLat) || isNaN(farmLng)) {
                console.error(`❌ Coordonnées NaN pour ${name}: lat=${farmLat}, lng=${farmLng}`);
                return;
            }
            
            // Vérifier que les coordonnées sont dans les limites de La Réunion
            if (farmLat < -22 || farmLat > -20 || farmLng < 55 || farmLng > 56) {
                console.warn(`⚠️ Coordonnées hors La Réunion pour ${name}: [${farmLat}, ${farmLng}]`);
            }
            
            try {
                const distanceData = await calculateRouteDistance(userLat, userLng, farmLat, farmLng);
                
                distances.push({
                    id: id,
                    name: name,
                    distance: distanceData.distance,
                    source: distanceData.source,
                    element: item
                });
                
                // Compter les sources
                if (distanceData.source === 'openrouteservice') {
                    orsCount++;
                } else {
                    fallbackCount++;
                }
                
                // Mettre à jour l'affichage dans la vue liste
                updateDistanceDisplay('distance-' + id, distanceData);
                
                // Mettre à jour l'affichage dans la vue grille
                updateDistanceDisplay('grid-distance-' + id, distanceData);
                
                // Stocker la distance dans l'élément pour le tri
                item.setAttribute('data-distance', distanceData.distance);
                
                // Faire la même chose pour l'élément correspondant dans la vue grille
                const gridItem = document.querySelector(`.grid-item[data-id="${id}"]`);
                if (gridItem) {
                    gridItem.setAttribute('data-distance', distanceData.distance);
                }
                
            } catch (error) {
                console.error(`❌ Erreur calcul distance pour ${name}:`, error);
                fallbackCount++;
            }
        });
        
        // Attendre que tous les calculs soient terminés
        await Promise.all(promises);
        
        console.log(`\n🏁 FIN calculs - ORS: ${orsCount}, Fallback: ${fallbackCount}`);
        
        // Trier les agriculteurs par distance
        distances.sort((a, b) => a.distance - b.distance);
        
        console.log('📊 Tri final par distance:');
        distances.forEach((d, i) => {
            console.log(`${i + 1}. ${d.name}: ${d.distance.toFixed(1)} km (${d.source})`);
        });
        
        // Réorganiser les éléments dans le DOM (vue liste)
        const listContainer = document.getElementById('farmer-list-' + moduleId);
        if (listContainer) {
            const sortedItems = distances.map(item => item.element.parentNode);
            sortedItems.forEach(li => {
                if (li) listContainer.appendChild(li);
            });
        }
        
        // Réorganiser les éléments dans le DOM (vue grille)
        const gridContainer = document.getElementById('farmer-grid-' + moduleId);
        if (gridContainer) {
            const gridItems = Array.from(document.querySelectorAll('.grid-item')).sort((a, b) => {
                return parseFloat(a.getAttribute('data-distance')) - parseFloat(b.getAttribute('data-distance'));
            });
            gridItems.forEach(item => {
                gridContainer.appendChild(item);
            });
        }
        
        // Réinitialiser le bouton
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            geolocateButton.disabled = false;
        }
        
        // Mettre à jour le message d'info
        if (distancesInfo) {
            distancesInfo.innerHTML = `<i class="fas fa-check-circle"></i> Distances recalculées depuis votre position actuelle.`;
        }
        
        // Afficher un message de succès avec statistiques
        showSuccessMessage({ ors: orsCount, fallback: fallbackCount });
    }
    
    // Fonction pour calculer les distances des résultats de recherche (version simplifiée)
    async function calculateSearchDistances(userLat, userLng) {
        console.log('🔍 Calcul distances pour résultats de recherche');
        // Version simplifiée pour se concentrer sur le problème principal
        if (!searchResults || searchResults.length === 0) return;
        
        // Logique similaire à calculateDistances() mais pour searchResults
        // ... (code similaire mais adapté pour les résultats de recherche)
    }
    
    // Simplification des autres fonctions pour se concentrer sur les distances
    function performSearch(query) {
        console.log('🔍 Recherche simplifiée:', query);
        // Version basique de la recherche
    }
    
    function loadGoogleMapsWithClusterer() {
        console.log('🗺️ Chargement Google Maps...');
        // Code Google Maps simplifié
    }
    
    // Mettre à jour les liens de pagination au chargement de la page
    updatePaginationLinks(activeView);
    
    // Précharger l'index de recherche
    setTimeout(function() {
        if (!searchIndex) {
            loadSearchIndex();
        }
    }, 500);
});