import L from 'leaflet';
import axios from 'axios';

// Configuration
const API_KEY = '5b3ce3597851110001cf62480c7ae4d074084cd6aefd8f3652182e26';
const ORS_API_URL = 'https://api.openrouteservice.org/v2/directions/driving-car';

// Coordonnées centrales de La Réunion
const REUNION_CENTER = [-21.115, 55.536];
const DEFAULT_ZOOM = 10;

// Éléments DOM
const startLatElement = document.getElementById('start-lat');
const startLngElement = document.getElementById('start-lng');
const destLatElement = document.getElementById('dest-lat');
const destLngElement = document.getElementById('dest-lng');
const calculateBtn = document.getElementById('calculate-btn');
const resultCard = document.getElementById('result-card');
const distanceElement = document.getElementById('distance');
const durationElement = document.getElementById('duration');
const errorElement = document.getElementById('error-message');
const useCurrentLocationBtn = document.getElementById('use-current-location');
const presetDestinations = document.getElementById('preset-destinations');

// Éléments pour l'édition manuelle des coordonnées
const editCoordinatesBtn = document.getElementById('edit-coordinates');
const manualCoordinatesDiv = document.getElementById('manual-coordinates');
const destinationCoordinatesDiv = document.getElementById('destination-coordinates');
const manualLatInput = document.getElementById('manual-lat');
const manualLngInput = document.getElementById('manual-lng');
const applyCoordinatesBtn = document.getElementById('apply-coordinates');
const cancelEditBtn = document.getElementById('cancel-edit');

// Initialisation de la carte
const map = L.map('map').setView(REUNION_CENTER, DEFAULT_ZOOM);

// Ajout de la couche de tuiles OpenStreetMap
L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
  attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
}).addTo(map);

// Marqueurs pour les positions
let startMarker = null;
let destinationMarker = null;
let routeLayer = null;

// État de l'application
let selectionMode = 'start'; // 'start' ou 'destination'

// Initialisation des marqueurs avec les valeurs par défaut
function initializeMarkers() {
  const startLat = parseFloat(startLatElement.textContent);
  const startLng = parseFloat(startLngElement.textContent);
  const destLat = parseFloat(destLatElement.textContent);
  const destLng = parseFloat(destLngElement.textContent);

  // Créer le marqueur de départ
  startMarker = L.marker([startLat, startLng], {
    draggable: true,
    icon: createCustomIcon('marker-start')
  }).addTo(map);
  
  startMarker.on('dragend', function(event) {
    const position = event.target.getLatLng();
    startLatElement.textContent = position.lat.toFixed(6);
    startLngElement.textContent = position.lng.toFixed(6);
    if (routeLayer) {
      map.removeLayer(routeLayer);
      routeLayer = null;
    }
  });

  // Créer le marqueur de destination
  destinationMarker = L.marker([destLat, destLng], {
    draggable: true,
    icon: createCustomIcon('marker-destination')
  }).addTo(map);
  
  destinationMarker.on('dragend', function(event) {
    const position = event.target.getLatLng();
    destLatElement.textContent = position.lat.toFixed(6);
    destLngElement.textContent = position.lng.toFixed(6);
    if (routeLayer) {
      map.removeLayer(routeLayer);
      routeLayer = null;
    }
  });
}

// Créer une icône personnalisée
function createCustomIcon(className) {
  return L.divIcon({
    className: className,
    iconSize: [20, 20],
    iconAnchor: [10, 10]
  });
}

// Gestionnaire de clic sur la carte
map.on('click', function(e) {
  const position = e.latlng;
  
  if (selectionMode === 'start') {
    if (startMarker) {
      startMarker.setLatLng(position);
    } else {
      startMarker = L.marker(position, {
        draggable: true,
        icon: createCustomIcon('marker-start')
      }).addTo(map);
      
      startMarker.on('dragend', function(event) {
        const pos = event.target.getLatLng();
        startLatElement.textContent = pos.lat.toFixed(6);
        startLngElement.textContent = pos.lng.toFixed(6);
        if (routeLayer) {
          map.removeLayer(routeLayer);
          routeLayer = null;
        }
      });
    }
    
    startLatElement.textContent = position.lat.toFixed(6);
    startLngElement.textContent = position.lng.toFixed(6);
    selectionMode = 'destination';
  } else {
    if (destinationMarker) {
      destinationMarker.setLatLng(position);
    } else {
      destinationMarker = L.marker(position, {
        draggable: true,
        icon: createCustomIcon('marker-destination')
      }).addTo(map);
      
      destinationMarker.on('dragend', function(event) {
        const pos = event.target.getLatLng();
        destLatElement.textContent = pos.lat.toFixed(6);
        destLngElement.textContent = pos.lng.toFixed(6);
        if (routeLayer) {
          map.removeLayer(routeLayer);
          routeLayer = null;
        }
      });
    }
    
    destLatElement.textContent = position.lat.toFixed(6);
    destLngElement.textContent = position.lng.toFixed(6);
    selectionMode = 'start';
  }
  
  // Supprimer l'itinéraire existant si présent
  if (routeLayer) {
    map.removeLayer(routeLayer);
    routeLayer = null;
  }
});

// Calculer la distance routière
async function calculateRouteDistance() {
  try {
    hideError();
    
    const startLat = parseFloat(startLatElement.textContent);
    const startLng = parseFloat(startLngElement.textContent);
    const destLat = parseFloat(destLatElement.textContent);
    const destLng = parseFloat(destLngElement.textContent);
    
    // Vérifier que les coordonnées sont valides
    if (isNaN(startLat) || isNaN(startLng) || isNaN(destLat) || isNaN(destLng)) {
      showError("Coordonnées invalides. Veuillez sélectionner des points valides sur la carte.");
      return;
    }
    
    // Vérifier que les coordonnées sont dans les limites de La Réunion
    if (!isInReunionBounds(startLat, startLng) || !isInReunionBounds(destLat, destLng)) {
      showError("Les coordonnées doivent être situées sur l'île de la Réunion.");
      return;
    }
    
    // Afficher un indicateur de chargement
    distanceElement.textContent = "Calcul en cours...";
    durationElement.textContent = "Calcul en cours...";
    resultCard.classList.remove('hidden');
    
    // Faire la requête à l'API
    const response = await axios.get(ORS_API_URL, {
      params: {
        start: `${startLng},${startLat}`,
        end: `${destLng},${destLat}`
      },
      headers: {
        'Accept': 'application/json, application/geo+json, application/gpx+xml, img/png; charset=utf-8',
        'Authorization': API_KEY,
        'Content-Type': 'application/json; charset=utf-8'
      }
    });
    
    // Traiter la réponse
    if (response.data && response.data.features && response.data.features.length > 0) {
      const route = response.data.features[0];
      const distance = (route.properties.segments[0].distance / 1000).toFixed(2); // Convertir en km
      const duration = formatDuration(route.properties.segments[0].duration);
      
      // Afficher les résultats
      distanceElement.textContent = `${distance} km`;
      durationElement.textContent = duration;
      
      // Dessiner l'itinéraire sur la carte
      drawRoute(route.geometry.coordinates);
    } else {
      showError("Impossible de calculer l'itinéraire. Veuillez vérifier vos points de départ et d'arrivée.");
    }
  } catch (error) {
    console.error("Erreur lors du calcul de l'itinéraire:", error);
    
    // Gérer les erreurs spécifiques de l'API
    if (error.response) {
      if (error.response.status === 401) {
        showError("Erreur d'authentification avec l'API. Veuillez vérifier votre clé API.");
      } else if (error.response.status === 404) {
        showError("Aucun itinéraire trouvé entre ces deux points. Ils sont peut-être inaccessibles par la route.");
      } else {
        showError(`Erreur du serveur: ${error.response.status} - ${error.response.statusText}`);
      }
    } else if (error.request) {
      showError("Impossible de contacter le serveur. Veuillez vérifier votre connexion internet.");
    } else {
      showError("Une erreur inattendue s'est produite. Veuillez réessayer.");
    }
  }
}

// Vérifier si les coordonnées sont dans les limites de La Réunion
function isInReunionBounds(lat, lng) {
  // Limites approximatives de La Réunion
  const REUNION_BOUNDS = {
    minLat: -21.4, maxLat: -20.8,
    minLng: 55.2, maxLng: 55.9
  };
  
  return (
    lat >= REUNION_BOUNDS.minLat && lat <= REUNION_BOUNDS.maxLat &&
    lng >= REUNION_BOUNDS.minLng && lng <= REUNION_BOUNDS.maxLng
  );
}

// Formater la durée en heures et minutes
function formatDuration(seconds) {
  const hours = Math.floor(seconds / 3600);
  const minutes = Math.floor((seconds % 3600) / 60);
  
  if (hours > 0) {
    return `${hours}h ${minutes}min`;
  } else {
    return `${minutes} minutes`;
  }
}

// Dessiner l'itinéraire sur la carte
function drawRoute(coordinates) {
  // Supprimer l'itinéraire existant si présent
  if (routeLayer) {
    map.removeLayer(routeLayer);
  }
  
  // Convertir les coordonnées au format attendu par Leaflet
  const latLngs = coordinates.map(coord => [coord[1], coord[0]]);
  
  // Créer une polyline pour l'itinéraire
  routeLayer = L.polyline(latLngs, {
    color: '#3498db',
    weight: 5,
    opacity: 0.7
  }).addTo(map);
  
  // Ajuster la vue de la carte pour montrer tout l'itinéraire
  map.fitBounds(routeLayer.getBounds(), { padding: [50, 50] });
}

// Afficher un message d'erreur
function showError(message) {
  errorElement.textContent = message;
  errorElement.classList.remove('hidden');
  resultCard.classList.add('hidden');
}

// Masquer le message d'erreur
function hideError() {
  errorElement.classList.add('hidden');
}

// Utiliser la géolocalisation du navigateur
function useCurrentLocation() {
  if (navigator.geolocation) {
    navigator.geolocation.getCurrentPosition(
      function(position) {
        const lat = position.coords.latitude;
        const lng = position.coords.longitude;
        
        // Vérifier si la position est dans les limites de La Réunion
        if (isInReunionBounds(lat, lng)) {
          startLatElement.textContent = lat.toFixed(6);
          startLngElement.textContent = lng.toFixed(6);
          
          if (startMarker) {
            startMarker.setLatLng([lat, lng]);
          } else {
            startMarker = L.marker([lat, lng], {
              draggable: true,
              icon: createCustomIcon('marker-start')
            }).addTo(map);
            
            startMarker.on('dragend', function(event) {
              const pos = event.target.getLatLng();
              startLatElement.textContent = pos.lat.toFixed(6);
              startLngElement.textContent = pos.lng.toFixed(6);
              if (routeLayer) {
                map.removeLayer(routeLayer);
                routeLayer = null;
              }
            });
          }
          
          map.setView([lat, lng], 12);
          selectionMode = 'destination';
        } else {
          showError("Votre position actuelle n'est pas sur l'île de la Réunion.");
        }
      },
      function(error) {
        let errorMessage;
        switch(error.code) {
          case error.PERMISSION_DENIED:
            errorMessage = "L'accès à la géolocalisation a été refusé.";
            break;
          case error.POSITION_UNAVAILABLE:
            errorMessage = "Les informations de localisation ne sont pas disponibles.";
            break;
          case error.TIMEOUT:
            errorMessage = "La demande de géolocalisation a expiré.";
            break;
          default:
            errorMessage = "Une erreur inconnue s'est produite lors de la géolocalisation.";
        }
        showError(errorMessage);
      }
    );
  } else {
    showError("La géolocalisation n'est pas prise en charge par votre navigateur.");
  }
}

// Gérer le changement de destination prédéfinie
function handlePresetDestinationChange() {
  const value = presetDestinations.value;
  if (value) {
    const [lat, lng] = value.split(',').map(parseFloat);
    destLatElement.textContent = lat.toFixed(6);
    destLngElement.textContent = lng.toFixed(6);
    
    if (destinationMarker) {
      destinationMarker.setLatLng([lat, lng]);
    } else {
      destinationMarker = L.marker([lat, lng], {
        draggable: true,
        icon: createCustomIcon('marker-destination')
      }).addTo(map);
      
      destinationMarker.on('dragend', function(event) {
        const pos = event.target.getLatLng();
        destLatElement.textContent = pos.lat.toFixed(6);
        destLngElement.textContent = pos.lng.toFixed(6);
        if (routeLayer) {
          map.removeLayer(routeLayer);
          routeLayer = null;
        }
      });
    }
    
    // Supprimer l'itinéraire existant si présent
    if (routeLayer) {
      map.removeLayer(routeLayer);
      routeLayer = null;
    }
  }
}

// Afficher le formulaire d'édition des coordonnées
function showCoordinatesEditor() {
  destinationCoordinatesDiv.classList.add('hidden');
  manualCoordinatesDiv.classList.remove('hidden');
  
  // Pré-remplir les champs avec les valeurs actuelles
  manualLatInput.value = destLatElement.textContent;
  manualLngInput.value = destLngElement.textContent;
}

// Masquer le formulaire d'édition des coordonnées
function hideCoordinatesEditor() {
  destinationCoordinatesDiv.classList.remove('hidden');
  manualCoordinatesDiv.classList.add('hidden');
}

// Appliquer les coordonnées manuelles
function applyManualCoordinates() {
  const lat = parseFloat(manualLatInput.value);
  const lng = parseFloat(manualLngInput.value);
  
  // Vérifier que les coordonnées sont valides
  if (isNaN(lat) || isNaN(lng)) {
    showError("Coordonnées invalides. Veuillez entrer des valeurs numériques.");
    return;
  }
  
  // Vérifier que les coordonnées sont dans les limites de La Réunion
  if (!isInReunionBounds(lat, lng)) {
    showError("Les coordonnées doivent être situées sur l'île de la Réunion.");
    return;
  }
  
  // Mettre à jour les coordonnées affichées
  destLatElement.textContent = lat.toFixed(6);
  destLngElement.textContent = lng.toFixed(6);
  
  // Mettre à jour le marqueur sur la carte
  if (destinationMarker) {
    destinationMarker.setLatLng([lat, lng]);
  } else {
    destinationMarker = L.marker([lat, lng], {
      draggable: true,
      icon: createCustomIcon('marker-destination')
    }).addTo(map);
    
    destinationMarker.on('dragend', function(event) {
      const pos = event.target.getLatLng();
      destLatElement.textContent = pos.lat.toFixed(6);
      destLngElement.textContent = pos.lng.toFixed(6);
      if (routeLayer) {
        map.removeLayer(routeLayer);
        routeLayer = null;
      }
    });
  }
  
  // Centrer la carte sur le nouveau point
  map.setView([lat, lng], 12);
  
  // Supprimer l'itinéraire existant si présent
  if (routeLayer) {
    map.removeLayer(routeLayer);
    routeLayer = null;
  }
  
  // Masquer l'éditeur
  hideCoordinatesEditor();
  hideError();
}

// Initialisation
document.addEventListener('DOMContentLoaded', function() {
  initializeMarkers();
  
  // Ajouter les écouteurs d'événements
  calculateBtn.addEventListener('click', calculateRouteDistance);
  useCurrentLocationBtn.addEventListener('click', useCurrentLocation);
  presetDestinations.addEventListener('change', handlePresetDestinationChange);
  
  // Écouteurs pour l'édition manuelle des coordonnées
  editCoordinatesBtn.addEventListener('click', showCoordinatesEditor);
  applyCoordinatesBtn.addEventListener('click', applyManualCoordinates);
  cancelEditBtn.addEventListener('click', hideCoordinatesEditor);
});
