<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_liste
 *
 * @copyright   Copyright (C) 2025 Votre Nom. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\HTML\HTMLHelper;

// Inclure le fichier d'aide du module de profil
require_once JPATH_SITE . '/modules/mod_agriculteur_profil/helper.php';

// Accès à la base de données
$db = Factory::getDbo();

// Récupérer les agriculteurs - on prend le formId des paramètres
$formId = $params->get('form_id', 1);

// Récupérer la catégorie filtrée depuis l'URL
$app = Factory::getApplication();
$filteredCategory = $app->input->getString('category', '');

// Requête pour récupérer tous les ID de soumission pour le formulaire
$query = $db->getQuery(true)
    ->select('s.SubmissionId')
    ->from($db->quoteName('#__rsform_submissions', 's'))
    ->where($db->quoteName('s.FormId') . ' = ' . $db->quote($formId))
    ->order($db->quoteName('s.DateSubmitted') . ' DESC');

$db->setQuery($query);
$submissionIds = $db->loadColumn();

// Initialiser le helper
$helper = new ModAgriculteurProfilHelper();

// Tableau pour stocker les agriculteurs et leurs distances
$agriculteurs = [];

// Position de l'utilisateur (à récupérer via HTML5 Geolocation API)
// Par défaut, coordonnées du centre de La Réunion
$userLat = -21.115141;
$userLng = 55.536384;

// Parcourir les soumissions et récupérer les données
foreach ($submissionIds as $submissionId) {
    $profileData = $helper->getProfileDataBySubmissionId($submissionId);
    
    if ($profileData) {
        // Filtrer par catégorie si une catégorie est demandée
        if (!empty($filteredCategory)) {
            // Vérifier si la catégorie demandée existe dans les catégories de l'agriculteur
            $categoriesMatch = false;
            
            // Rechercher dans toutes les catégories possibles
            foreach (['categories', 'type_agriculture', 'type_elevage', 'autres_activites'] as $categoryField) {
                if (isset($profileData[$categoryField]) && is_array($profileData[$categoryField])) {
                    foreach ($profileData[$categoryField] as $cat) {
                        // Comparer la catégorie sans distinction de casse
                        if (strcasecmp(trim($cat), trim($filteredCategory)) === 0) {
                            $categoriesMatch = true;
                            break 2; // Sortir des deux boucles
                        }
                    }
                }
            }
            
            // Vérifier aussi dans le statut
            if (isset($profileData['status']) && strcasecmp(trim($profileData['status']), trim($filteredCategory)) === 0) {
                $categoriesMatch = true;
            }
            
            // Si aucune correspondance, passer à l'agriculteur suivant
            if (!$categoriesMatch) {
                continue;
            }
        }
        
        // Calculer la distance (à vol d'oiseau initialement)
        $distance = 0;
        
        if (!empty($profileData['geolocalisation'])) {
            list($agriLat, $agriLng) = explode(',', $profileData['geolocalisation']);
            
            // Distance à vol d'oiseau (formule Haversine)
            $distance = calculateDistance($userLat, $userLng, (float)$agriLat, (float)$agriLng);
        }
        
        // Récupérer l'URL de l'image de couverture
        $coverImageUrl = !empty($profileData['photo_couverture-petit-format_url']) 
            ? $profileData['photo_couverture_petit_format_url'] 
            : $helper->getUploadPath($profileData['photo_couverture_petit_format']);
            
        // Si pas d'image de couverture, utiliser une image par défaut
        if (empty($coverImageUrl)) {
            $coverImageUrl = Uri::root() . 'media/mod_agriculteur_liste/images/default-cover.jpg';
        }
        
        // Ajouter les données à notre tableau
        $agriculteurs[] = [
            'id' => $submissionId,
            'societe' => $profileData['societe'],
            'nom' => $profileData['nom'],
            'prenom' => $profileData['prenom'],
            'distance' => $distance,
            'image' => $coverImageUrl,
            'coordinates' => $profileData['geolocalisation']
        ];
    }
}

// Trier les agriculteurs par distance
usort($agriculteurs, function($a, $b) {
    return $a['distance'] <=> $b['distance'];
});

// Fonction pour calculer la distance à vol d'oiseau
function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371; // Rayon de la Terre en km
    
    $latFrom = deg2rad($lat1);
    $lonFrom = deg2rad($lon1);
    $latTo = deg2rad($lat2);
    $lonTo = deg2rad($lon2);
    
    $latDelta = $latTo - $latFrom;
    $lonDelta = $lonTo - $lonFrom;
    
    $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) + 
        cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));
        
    return $angle * $earthRadius;
}

// Charger les styles et scripts nécessaires
$document = Factory::getDocument();

// Chargement des styles CSS
if ($params->get('load_styles', 1)) {
    $cssFile = 'media/mod_agriculteur_liste/css/style.css';
    if (file_exists(JPATH_SITE . '/' . $cssFile)) {
        HTMLHelper::_('stylesheet', $cssFile, ['version' => 'auto']);
    } else {
        // Les styles CSS existants...
    }
}

// Inclure le template d'affichage
require ModuleHelper::getLayoutPath('mod_agriculteur_liste', $params->get('layout', 'default'));