<?php
/**
 * Script CLI simplifié pour générer l'index de recherche des agriculteurs
 * 
 * Usage: php generate_search_index.php
 */

// Configuration de la base de données - à adapter selon votre configuration
$db_config = [
    'host' => 'localhost',       // Généralement localhost ou 127.0.0.1
    'user' => 'labiolela_site',  // Nom d'utilisateur de la base de données
    'pass' => 'xr650r@852',                // Mot de passe de la base de données (à remplir)
    'name' => 'labiolela_site',  // Nom de la base de données
    'prefix' => 'wqkom_'         // Préfixe des tables
];

// Activer l'affichage des erreurs
ini_set('display_errors', 1);
ini_set('log_errors', 1);
error_reporting(E_ALL);

// Répertoire racine (parent du dossier cli)
$root_path = dirname(dirname(__FILE__));

// URL racine du site (à adapter selon votre configuration)
$site_url = 'https://labiolela.re';

// Chemin du fichier d'index
$index_path = $root_path . '/media/mod_agriculteur_liste/search-index.json';
$media_path = $root_path . '/media/mod_agriculteur_liste';

// Créer le dossier media s'il n'existe pas
if (!file_exists($media_path)) {
    mkdir($media_path, 0755, true);
}

// Créer le dossier images s'il n'existe pas
$images_path = $media_path . '/images';
if (!file_exists($images_path)) {
    mkdir($images_path, 0755, true);
}

// Créer le dossier de logs s'il n'existe pas
$logs_path = $root_path . '/logs';
if (!file_exists($logs_path)) {
    mkdir($logs_path, 0755, true);
}

ini_set('error_log', $logs_path . '/search_index_errors.log');

echo "Début de la génération de l'index de recherche.\n";
echo "Chemin d'index: " . $index_path . "\n";

// Vérifier si le dossier media existe
if (!file_exists($media_path)) {
    echo "Création du dossier: $media_path\n";
    mkdir($media_path, 0755, true);
}

echo "Le dossier existe: " . (file_exists($media_path) ? 'Oui' : 'Non') . "\n";
echo "Dossier accessible en écriture: " . (is_writable($media_path) ? 'Oui' : 'Non') . "\n";

try {
    // Connexion à la base de données
    echo "Connexion à la base de données...\n";
    $mysqli = new mysqli($db_config['host'], $db_config['user'], $db_config['pass'], $db_config['name']);

    if ($mysqli->connect_error) {
        throw new Exception("Erreur de connexion à la base de données: " . $mysqli->connect_error);
    }

    $mysqli->set_charset("utf8");
    echo "Connexion réussie.\n";

    // Préparation des noms de tables avec préfixe
    $submissions_table = $db_config['prefix'] . 'rsform_submissions';
    $submission_values_table = $db_config['prefix'] . 'rsform_submission_values';

    // Vérifier si les tables existent
    $tables_result = $mysqli->query("SHOW TABLES LIKE '" . $submissions_table . "'");
    if ($tables_result->num_rows == 0) {
        throw new Exception("La table $submissions_table n'existe pas.");
    }

    $tables_result = $mysqli->query("SHOW TABLES LIKE '" . $submission_values_table . "'");
    if ($tables_result->num_rows == 0) {
        throw new Exception("La table $submission_values_table n'existe pas.");
    }

    echo "Tables trouvées.\n";

    // ID du formulaire RSForm - d'après l'image, formId est 8
    $form_id = 8;

    // Récupérer les soumissions
    $query = "SELECT * FROM $submissions_table WHERE FormId = $form_id";
    $result = $mysqli->query($query);

    if (!$result) {
        throw new Exception("Erreur lors de la requête SQL: " . $mysqli->error);
    }

    $submissions = [];
    while ($row = $result->fetch_assoc()) {
        $submissions[] = $row;
    }

    echo "Nombre de soumissions trouvées: " . count($submissions) . "\n";

    // Récupérer les données des champs pour chaque soumission
    $farmers = [];
    foreach ($submissions as $submission) {
        $submission_id = $submission['SubmissionId'];
        
        // Requête pour récupérer les valeurs de soumission
        $query = "SELECT FieldName, FieldValue FROM $submission_values_table WHERE SubmissionId = $submission_id";
        $fields_result = $mysqli->query($query);

        if (!$fields_result) {
            echo "Erreur lors de la requête SQL pour les champs de la soumission #$submission_id: " . $mysqli->error . "\n";
            continue;
        }

        // Créer un tableau avec les données de l'agriculteur
        $farmer = [
            'id' => $submission_id,
            'date_creation' => $submission['DateSubmitted'],
            'fields' => []
        ];
        
        // Parcourir tous les champs et les ajouter au tableau
        while ($field = $fields_result->fetch_assoc()) {
            $name = $field['FieldName'];
            $value = $field['FieldValue'];
            
            // Décodage des valeurs JSON si nécessaire
            if (substr($value, 0, 1) === '[' || substr($value, 0, 1) === '{') {
                $decoded_value = json_decode($value, true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $value = $decoded_value;
                }
            }
            
            // Traitement spécial pour les champs à valeurs multiples (comme les cases à cocher)
            if (is_string($value) && strpos($value, '|') !== false) {
                $value = explode('|', $value);
                // Nettoyer les valeurs vides
                $value = array_filter($value, function($item) {
                    return !empty(trim($item));
                });
            }
            
            $farmer['fields'][$name] = $value;
        }
        
        $farmers[] = $farmer;
    }
    
    echo "Données extraites pour " . count($farmers) . " agriculteurs.\n";
    
    // Préparer l'index
    $index = [
        'version' => date('Y-m-d') . '_001',
        'lastUpdate' => date('c'), // Format ISO 8601
        'entries' => []
    ];
    
    // Mappage des noms de champs RSForm vers les noms d'index
    // Mis à jour selon les images partagées
    $field_mapping = [
        'societe' => 'societe',
        'nom' => 'nom',
        'prenom' => 'prenom',
        'description' => 'information',
        'adresse' => 'adresse',
        'code_postal' => 'code-postal',
        'commune' => 'ville',
        'telephone' => 'telephone',
        'email' => 'Email',
        'site_web' => 'Site-internet',
        'produits' => 'produits-semaine',
        'labels' => '',
        'certifications' => '',
        'modes_vente' => 'vente',
        'categorie' => 'categories',
        'type_agriculture' => 'type-agriculture',
        'horaires' => 'horaires-reception',
        'jours' => 'jours-reception',
        'latitude' => '', // Nous allons extraire du champ geolocalisation
        'longitude' => '', // Nous allons extraire du champ geolocalisation
        'image_url' => 'photo_couverture_url',
        'facebook' => 'facebook'
    ];
    
    // Logger quelques noms de champs pour débogage
    echo "Exemples de noms de champs dans les données:\n";
    if (!empty($farmers)) {
        $fields_sample = array_keys($farmers[0]['fields']);
        echo implode(", ", array_slice($fields_sample, 0, 10)) . "\n";
    }
    
    // Pour chaque agriculteur, créer une entrée dans l'index
    foreach ($farmers as $farmer) {
        $fields = $farmer['fields'];
        
        // Extraire les coordonnées du champ geolocalisation si présent
        $coordinates = '';
        if (!empty($fields['geolocalisation'])) {
            $coordinates = trim($fields['geolocalisation']);
        }
        
        // Recherche de texte améliorée
        $search_parts = [];
        
        // Traiter chaque champ pour le texte de recherche
        foreach ($fields as $field_name => $value) {
            // Ajouter chaque valeur au texte de recherche
            if (is_array($value)) {
                foreach ($value as $sub_value) {
                    if (is_string($sub_value) || is_numeric($sub_value)) {
                        $search_parts[] = trim((string)$sub_value);
                    }
                }
            } elseif (is_string($value) || is_numeric($value)) {
                $search_parts[] = trim((string)$value);
            }
        }
        
        // Assembler le texte de recherche
        $search_text = implode(' ', $search_parts);
        $search_text = preg_replace('/\s+/', ' ', $search_text); // Remplacer les espaces multiples
        $search_text = strtolower(trim($search_text)); // Tout en minuscules et trim
        
        // Fonction pour récupérer un champ avec le mapping
        $get_mapped_field = function($key) use ($fields, $field_mapping) {
            $field_name = $field_mapping[$key] ?? $key;
            if (empty($field_name)) {
                return '';
            }
            return $fields[$field_name] ?? '';
        };
        
        // Convertir les champs en tableaux si nécessaire
        $get_array_field = function($key) use ($get_mapped_field) {
            $value = $get_mapped_field($key);
            if (is_string($value)) {
                return [$value];
            } elseif (is_array($value)) {
                return $value;
            }
            return [];
        };
        
        // Fusionner les horaires et jours pour une information complète
        $hours = '';
        if (!empty($fields['jours-reception'])) {
            $hours .= $fields['jours-reception'] . ' ';
        }
        if (!empty($fields['horaires-reception'])) {
            $hours .= $fields['horaires-reception'];
        }
        
        // Fonctions pour traiter l'URL de l'image
        $get_image_url = function() use ($fields, $root_path, $site_url) {
            // Essayer différentes clés pour les images
            $image_keys = ['photo_couverture_url', 'photo-couverture', 'photo-couverture-petit-format', 'photo_couverture_url'];
            
            foreach ($image_keys as $key) {
                if (!empty($fields[$key])) {
                    $photo_path = $fields[$key];
                    
                    // Si c'est déjà une URL, la renvoyer directement
                    if (strpos($photo_path, 'http') === 0) {
                        return $photo_path;
                    }
                    
                    // Convertir le chemin absolu du serveur en URL relative
                    if (strpos($photo_path, '/home/labiolela/public_html/') === 0) {
                        return str_replace('/home/labiolela/public_html/', '/', $photo_path);
                    }
                    
                    // Si c'est un chemin relatif, le renvoyer tel quel
                    return $photo_path;
                }
            }
            
            // Si aucune image n'est trouvée, renvoyer une chaîne vide
            return '';
        };
        
        // Créer l'entrée d'index
        $entry = [
            'id' => $farmer['id'],
            'name' => $get_mapped_field('societe'),
            'owner' => $get_mapped_field('prenom') . ' ' . $get_mapped_field('nom'),
            'description' => $get_mapped_field('description'),
            'address' => $get_mapped_field('adresse'),
            'postal_code' => $get_mapped_field('code_postal'),
            'location' => $get_mapped_field('commune'),
            'phone' => $get_mapped_field('telephone'),
            'email' => $get_mapped_field('email'),
            'website' => $get_mapped_field('site_web'),
            'products' => $get_array_field('produits'),
            'categories' => $get_array_field('categorie'),
            'type_agriculture' => $get_mapped_field('type_agriculture'),
            'sales_methods' => $get_array_field('modes_vente'),
            'hours' => $hours,
            'coordinates' => $coordinates,
            'image' => $get_image_url(),
            'facebook' => $get_mapped_field('facebook'),
            'searchableText' => $search_text,
            'distance' => 0 // Distance initiale à 0
        ];
        
        // Ajouter à l'index
        $index['entries'][] = $entry;
    }
    
    // Convertir en JSON
    $json_index = json_encode($index, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
    // Sauvegarder le fichier
    $result = file_put_contents($index_path, $json_index);
    
    // Afficher des statistiques
    if ($result) {
        echo "Index de recherche généré avec succès.\n";
        echo "Nombre d'entrées : " . count($index['entries']) . "\n";
        echo "Taille du fichier : " . round(filesize($index_path) / 1024, 2) . " Ko\n";
        echo "Fichier sauvegardé : " . $index_path . "\n";
    } else {
        echo "ERREUR : Impossible d'écrire le fichier d'index!\n";
        echo "Vérifiez les permissions d'écriture dans le dossier : " . $media_path . "\n";
    }
    
    // Fermer la connexion
    $mysqli->close();
    
} catch (Exception $e) {
    echo "Erreur d'exécution : " . $e->getMessage() . "\n";
    if (isset($e->getFile)) {
        echo "Fichier : " . $e->getFile() . " à la ligne " . $e->getLine() . "\n";
    }
    exit(1);
}