class AgriculteurModule {
    constructor(options) {
        // Configuration initiale
        this.options = {
            agriculteurs: options.agriculteurs || [],
            currentView: 'list',
            userLocation: null
        };

        // Initialiser les éléments DOM
        this.initElements();

        // Configurer les événements
        this.setupEventListeners();
    }

    initElements() {
        // Sélectionner les éléments principaux
        this.listContainer = document.getElementById('agriculteurs-list-container');
        this.mapContainer = document.getElementById('agriculteurs-map-container');
        
        // Boutons de navigation
        this.listViewToggle = document.getElementById('list-view-toggle');
        this.gridViewToggle = document.getElementById('grid-view-toggle');
        this.mapViewToggle = document.getElementById('map-view-toggle');
    }

    setupEventListeners() {
        // Gestionnaires d'événements pour les boutons de vue
        this.listViewToggle.addEventListener('click', () => this.changeView('list'));
        this.gridViewToggle.addEventListener('click', () => this.changeView('grid'));
        this.mapViewToggle.addEventListener('click', () => this.changeView('map'));

        // Géolocalisation
        this.setupGeolocation();
    }

    changeView(viewType) {
        // Mettre à jour la vue active
        this.listContainer.style.display = viewType === 'list' ? 'block' : 'none';
        this.mapContainer.style.display = viewType === 'map' ? 'block' : 'none';

        // Mise à jour de la vue grille (à implémenter)
        if (viewType === 'grid') {
            this.switchToGridView();
        }

        // Initialiser la carte si nécessaire
        if (viewType === 'map') {
            this.initMap();
        }

        // Mettre à jour l'état
        this.options.currentView = viewType;
    }

    setupGeolocation() {
        // Bouton de géolocalisation
        const geolocateButton = document.createElement('button');
        geolocateButton.textContent = 'Localiser';
        geolocateButton.addEventListener('click', () => this.getUserLocation());

        // Ajouter le bouton quelque part dans l'interface
        document.querySelector('.navigation-controls').appendChild(geolocateButton);
    }

    getUserLocation() {
        if ('geolocation' in navigator) {
            navigator.geolocation.getCurrentPosition(
                (position) => {
                    // Stocker la position de l'utilisateur
                    this.options.userLocation = {
                        lat: position.coords.latitude,
                        lng: position.coords.longitude
                    };

                    // Calculer les distances
                    this.calculateDistances();
                },
                (error) => {
                    console.error('Erreur de géolocalisation:', error);
                    alert('Impossible de récupérer votre position.');
                }
            );
        } else {
            alert('La géolocalisation n\'est pas supportée par votre navigateur.');
        }
    }

    calculateDistances() {
        // Calculer la distance entre deux points (Haversine)
        const haversineDistance = (lat1, lon1, lat2, lon2) => {
            const R = 6371; // Rayon de la Terre en km
            const dLat = this.toRadians(lat2 - lat1);
            const dLon = this.toRadians(lon2 - lon1);
            const a = 
                Math.sin(dLat/2) * Math.sin(dLat/2) +
                Math.cos(this.toRadians(lat1)) * Math.cos(this.toRadians(lat2)) * 
                Math.sin(dLon/2) * Math.sin(dLon/2);
            const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
            return R * c;
        };

        // Mettre à jour les distances pour chaque agriculteur
        this.options.agriculteurs.forEach(agriculteur => {
            if (this.options.userLocation) {
                const [lat, lng] = agriculteur.coordinates.split(',').map(parseFloat);
                agriculteur.distance = haversineDistance(
                    this.options.userLocation.lat, 
                    this.options.userLocation.lng, 
                    lat, 
                    lng
                );
            }
        });

        // Trier les agriculteurs par distance
        this.options.agriculteurs.sort((a, b) => a.distance - b.distance);

        // Mettre à jour l'affichage
        this.updateAgriculteursList();
    }

    toRadians(degrees) {
        return degrees * (Math.PI / 180);
    }

    updateAgriculteursList() {
        // Mettre à jour la liste des agriculteurs
        const container = this.listContainer;
        container.innerHTML = ''; // Vider le conteneur

        this.options.agriculteurs.forEach(agriculteur => {
            const itemElement = document.createElement('div');
            itemElement.classList.add('agriculteur-item');
            itemElement.innerHTML = `
                <img src="${agriculteur.image}" alt="${agriculteur.societe}">
                <div class="agriculteur-details">
                    <h3>${agriculteur.societe}</h3>
                    <p>${agriculteur.prenom} ${agriculteur.nom}</p>
                    <span class="distance">${agriculteur.distance ? agriculteur.distance.toFixed(1) : 'N/A'} km</span>
                </div>
            `;
            container.appendChild(itemElement);
        });
    }

    initMap() {
        // Vérifier si Google Maps est chargé
        if (typeof google === 'undefined') {
            this.loadGoogleMapsScript();
            return;
        }

        // Créer la carte
        const mapElement = document.getElementById('agriculteurs-map');
        const map = new google.maps.Map(mapElement, {
            center: { lat: -21.115141, lng: 55.536384 }, // Centre de La Réunion
            zoom: 10
        });

        // Ajouter des marqueurs pour chaque agriculteur
        this.options.agriculteurs.forEach(agriculteur => {
            const [lat, lng] = agriculteur.coordinates.split(',').map(parseFloat);
            new google.maps.Marker({
                position: { lat, lng },
                map: map,
                title: agriculteur.societe
            });
        });
    }

    loadGoogleMapsScript() {
        // Charger le script Google Maps
        const script = document.createElement('script');
        script.src = `https://maps.googleapis.com/maps/api/js?key=VOTRE_CLE_API&callback=initMap`;
        script.async = true;
        script.defer = true;
        
        // Ajouter une méthode globale pour initialiser la carte
        window.initMap = () => this.initMap();
        
        document.head.appendChild(script);
    }
}