<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_liste
 *
 * @copyright   (C) 2023 Open Source Matters, Inc. <https://www.joomla.org>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\Content\Site\Helper\RouteHelper;

/**
 * Helper for mod_agriculteur_liste
 *
 * @since  1.0
 */
class ModAgriculteurListeHelper
{
    /**
     * Récupère une liste d'agriculteurs
     *
     * @param   \Joomla\Registry\Registry  &$params  object holding the models parameters
     *
     * @return  mixed
     */
    public static function getList(&$params)
    {
        $app = Factory::getApplication();
        $db = Factory::getDbo();
        $user = Factory::getUser();
        $groups = $user->getAuthorisedViewLevels();
        $catid = $params->get('catid');
        $items = array();

        // Requête pour récupérer les agriculteurs
        $query = $db->getQuery(true)
            ->select(
                [
                    'a.id',
                    'a.title',
                    'a.alias',
                    'a.introtext',
                    'a.fulltext',
                    'a.checked_out',
                    'a.checked_out_time',
                    'a.catid',
                    'a.created',
                    'a.created_by',
                    'a.created_by_alias',
                    'a.modified',
                    'a.modified_by',
                    'a.publish_up',
                    'a.publish_down',
                    'a.images',
                    'a.urls',
                    'a.attribs',
                    'a.metadata',
                    'a.metakey',
                    'a.metadesc',
                    'a.access',
                    'a.hits',
                    'a.featured',
                    'a.language',
                ]
            )
            ->from('#__content AS a')
            ->where('a.catid = ' . (int) $catid)
            ->where('a.state = 1')
            ->where('a.access IN (' . implode(',', $groups) . ')');

        // Filtrer par langue
        $languageFilter = $app->getLanguageFilter();
        if ($languageFilter) {
            $query->where('a.language IN (' . $db->quote(Factory::getLanguage()->getTag()) . ',' . $db->quote('*') . ')');
        }

        $query->order('a.created DESC');

        $db->setQuery($query);
        $items = $db->loadObjectList();

        // Traitement des items
        foreach ($items as &$item) {
            $item->slug = $item->id . ':' . $item->alias;
            $item->catslug = $item->catid . ':' . $item->category_alias;
            $item->link = RouteHelper::getArticleRoute($item->slug, $item->catid, $item->language);

            // Récupération des champs personnalisés
            $item->jcfields = self::getCustomFields($item->id);
            
            // Traitement des champs personnalisés pour faciliter l'accès
            $item->custom_fields = [];
            foreach ($item->jcfields as $field) {
                $item->custom_fields[$field->name] = $field;
            }
            
            // Récupération des données RSForm associées si disponibles
            $item->rsform_data = self::getRSFormData($item->id);
        }

        return $items;
    }

    /**
     * Récupère les champs personnalisés d'un article
     *
     * @param   int  $articleId  ID de l'article
     *
     * @return  array
     */
    public static function getCustomFields($articleId)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        
        $query->select('f.id, f.name, f.label, f.type, f.params, v.value')
            ->from('#__fields AS f')
            ->join('INNER', '#__fields_values AS v ON v.field_id = f.id')
            ->where('v.item_id = ' . (int) $articleId)
            ->where('f.context = ' . $db->quote('com_content.article'))
            ->where('f.state = 1')
            ->order('f.ordering ASC');
        
        $db->setQuery($query);
        $fields = $db->loadObjectList();
        
        // Traitement des paramètres JSON
        foreach ($fields as &$field) {
            $field->params = json_decode($field->params);
        }
        
        return $fields;
    }
    
    /**
     * Récupère les données RSForm associées à un article
     *
     * @param   int  $articleId  ID de l'article
     *
     * @return  array
     */
    public static function getRSFormData($articleId)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        
        // Récupération de l'ID de formulaire RSForm associé à l'article (si configuré)
        $formId = self::getAssociatedRSFormId($articleId);
        
        if (!$formId) {
            return [];
        }
        
        // Récupération des données du formulaire RSForm
        $query->select('s.SubmissionId, s.DateSubmitted, s.UserIp, d.FieldName, d.FieldValue')
            ->from('#__rsform_submissions AS s')
            ->join('INNER', '#__rsform_submission_values AS d ON d.SubmissionId = s.SubmissionId')
            ->where('s.FormId = ' . (int) $formId)
            ->order('s.DateSubmitted DESC');
        
        $db->setQuery($query);
        $rsformData = $db->loadObjectList();
        
        // Organiser les données par soumission
        $submissions = [];
        foreach ($rsformData as $data) {
            if (!isset($submissions[$data->SubmissionId])) {
                $submissions[$data->SubmissionId] = [
                    'id' => $data->SubmissionId,
                    'date' => $data->DateSubmitted,
                    'ip' => $data->UserIp,
                    'fields' => []
                ];
            }
            
            $submissions[$data->SubmissionId]['fields'][$data->FieldName] = $data->FieldValue;
        }
        
        return array_values($submissions);
    }
    
    /**
     * Récupère l'ID du formulaire RSForm associé à un article
     *
     * @param   int  $articleId  ID de l'article
     *
     * @return  int|null
     */
    private static function getAssociatedRSFormId($articleId)
    {
        // Cette méthode doit être adaptée selon la façon dont vous associez les formulaires RSForm aux articles
        // Par exemple, via un champ personnalisé, une table de liaison, etc.
        
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        
        // Exemple: recherche dans un champ personnalisé nommé "rsform_id"
        $query->select('v.value')
            ->from('#__fields AS f')
            ->join('INNER', '#__fields_values AS v ON v.field_id = f.id')
            ->where('v.item_id = ' . (int) $articleId)
            ->where('f.name = ' . $db->quote('rsform_id'))
            ->where('f.context = ' . $db->quote('com_content.article'));
        
        $db->setQuery($query);
        $formId = $db->loadResult();
        
        return $formId ? (int) $formId : null;
    }

    /**
     * Calcule la distance routière entre deux points GPS
     *
     * @param   float  $startLat   Latitude du point de départ
     * @param   float  $startLng   Longitude du point de départ
     * @param   float  $destLat    Latitude du point d'arrivée
     * @param   float  $destLng    Longitude du point d'arrivée
     *
     * @return  array  Distance en km et durée en minutes
     */
    public static function calculateRouteDistance($startLat, $startLng, $destLat, $destLng)
    {
        // Configuration de l'API OpenRouteService
        $apiKey = '5b3ce3597851110001cf62480c7ae4d074084cd6aefd8f3652182e26';
        $apiUrl = 'https://api.openrouteservice.org/v2/directions/driving-car';
        
        // Vérifier que les coordonnées sont valides
        if (!is_numeric($startLat) || !is_numeric($startLng) || !is_numeric($destLat) || !is_numeric($destLng)) {
            return ['distance' => 0, 'duration' => 0];
        }
        
        // Vérifier que les coordonnées sont dans les limites de La Réunion
        if (!self::isInReunionBounds($startLat, $startLng) || !self::isInReunionBounds($destLat, $destLng)) {
            return ['distance' => 0, 'duration' => 0];
        }
        
        // Préparer les paramètres de la requête
        $params = [
            'api_key' => $apiKey,
            'start' => $startLng . ',' . $startLat,
            'end' => $destLng . ',' . $destLat
        ];
        
        // Construire l'URL avec les paramètres
        $url = $apiUrl . '?' . http_build_query($params);
        
        // Initialiser cURL
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Accept: application/json, application/geo+json, application/gpx+xml, img/png; charset=utf-8',
            'Authorization: ' . $apiKey,
            'Content-Type: application/json; charset=utf-8'
        ]);
        
        // Exécuter la requête
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        // Traiter la réponse
        if ($httpCode == 200) {
            $data = json_decode($response, true);
            
            if (isset($data['features']) && !empty($data['features'])) {
                $route = $data['features'][0];
                $distance = $route['properties']['segments'][0]['distance'] / 1000; // Convertir en km
                $duration = $route['properties']['segments'][0]['duration'] / 60; // Convertir en minutes
                
                return [
                    'distance' => round($distance, 2),
                    'duration' => round($duration)
                ];
            }
        }
        
        // En cas d'erreur, retourner des valeurs par défaut
        return ['distance' => 0, 'duration' => 0];
    }
    
    /**
     * Vérifie si les coordonnées sont dans les limites de La Réunion
     *
     * @param   float  $lat  Latitude
     * @param   float  $lng  Longitude
     *
     * @return  bool
     */
    public static function isInReunionBounds($lat, $lng)
    {
        // Limites approximatives de La Réunion
        $reunionBounds = [
            'minLat' => -21.4, 'maxLat' => -20.8,
            'minLng' => 55.2, 'maxLng' => 55.9
        ];
        
        return (
            $lat >= $reunionBounds['minLat'] && $lat <= $reunionBounds['maxLat'] &&
            $lng >= $reunionBounds['minLng'] && $lng <= $reunionBounds['maxLng']
        );
    }
    
    /**
     * Formate la durée en heures et minutes
     *
     * @param   int  $minutes  Durée en minutes
     *
     * @return  string
     */
    public static function formatDuration($minutes)
    {
        if ($minutes < 1) {
            return "< 1 min";
        }
        
        $hours = floor($minutes / 60);
        $mins = $minutes % 60;
        
        if ($hours > 0) {
            return $hours . "h " . ($mins > 0 ? $mins . "min" : "");
        } else {
            return $mins . " min";
        }
    }
}
