<?php
// Fichier : get-distance.php
// But : interroger Google Maps API côté serveur pour calculer la distance routière entre 2 points

// Définir que la réponse sera en JSON
header('Content-Type: application/json');

// Activer l'affichage des erreurs pour le débogage
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Créer un fichier de log pour le débogage
$logFile = __DIR__ . '/distance_api.log';
function writeLog($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$timestamp] $message" . PHP_EOL, FILE_APPEND);
}

writeLog("=== Nouvelle requête API ===");

// Vérifie que les paramètres start et end sont fournis
if (!isset($_GET['start']) || !isset($_GET['end'])) {
    $error = 'Missing parameters: start and end are required';
    writeLog("ERREUR: $error");
    echo json_encode(['error' => $error]);
    exit;
}

// Récupère les coordonnées depuis l'URL
$start = $_GET['start']; // Format attendu : "longitude,latitude"
$end   = $_GET['end'];   // Format attendu : "longitude,latitude"

writeLog("Paramètres reçus - start: $start, end: $end");

// Vérifier que les coordonnées sont au bon format (simplifié pour accepter plus de formats)
if (!preg_match('/^[-+]?([0-9]*\.?[0-9]+),\s*[-+]?([0-9]*\.?[0-9]+)$/', $start) ||
    !preg_match('/^[-+]?([0-9]*\.?[0-9]+),\s*[-+]?([0-9]*\.?[0-9]+)$/', $end)) {
    $error = 'Invalid coordinates format. Expected: "longitude,latitude"';
    writeLog("ERREUR: $error - start: $start, end: $end");
    echo json_encode(['error' => $error]);
    exit;
}

// Clé API Google Maps
$apiKey = 'AIzaSyAjsOoV0528RtrLFvylBrsypk0Z3qik5RI';

try {
    // Extraire les coordonnées pour les formater correctement
    list($startLng, $startLat) = explode(',', $start);
    list($endLng, $endLat) = explode(',', $end);
    
    // Convertir en nombres à virgule flottante
    $startLng = (float)$startLng;
    $startLat = (float)$startLat;
    $endLng = (float)$endLng;
    $endLat = (float)$endLat;
    
    writeLog("Coordonnées formatées - start: [$startLng, $startLat], end: [$endLng, $endLat]");
    
    // Vérifier que les coordonnées sont dans des plages valides
    if ($startLat < -90 || $startLat > 90 || $endLat < -90 || $endLat > 90 ||
        $startLng < -180 || $startLng > 180 || $endLng < -180 || $endLng > 180) {
        $error = 'Coordinates out of valid range';
        writeLog("ERREUR: $error - start: [$startLng, $startLat], end: [$endLng, $endLat]");
        echo json_encode(['error' => $error]);
        exit;
    }
    
    // Formater les coordonnées pour l'API Google Maps
    // Google Maps attend le format "latitude,longitude" (inverse de OpenRouteService)
    $originCoords = "$startLat,$startLng";
    $destinationCoords = "$endLat,$endLng";
    
    // Construction de l'URL de l'API Google Maps Distance Matrix
    $url = "https://maps.googleapis.com/maps/api/distancematrix/json?";
    $url .= "origins=" . urlencode($originCoords);
    $url .= "&destinations=" . urlencode($destinationCoords);
    $url .= "&mode=driving"; // Mode de transport (driving, walking, bicycling, transit)
    $url .= "&language=fr"; // Langue des résultats
    $url .= "&key=" . $apiKey;
    
    writeLog("URL de l'API Google Maps: " . preg_replace('/key=([^&]+)/', 'key=HIDDEN', $url));
    
    // Initialise une requête cURL
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 20);
    
    // Option pour voir les détails de la requête
    curl_setopt($ch, CURLOPT_VERBOSE, true);
    $verbose = fopen('php://temp', 'w+');
    curl_setopt($ch, CURLOPT_STDERR, $verbose);
    
    writeLog("Envoi de la requête à l'API Google Maps...");
    $response = curl_exec($ch);
    
    // Récupérer les informations de débogage
    rewind($verbose);
    $verboseLog = stream_get_contents($verbose);
    writeLog("Détails cURL: " . str_replace("\n", " | ", $verboseLog));
    
    // Gère les erreurs cURL
    if (curl_errno($ch)) {
        $error = 'cURL Error: ' . curl_error($ch);
        writeLog("ERREUR: $error");
        echo json_encode(['error' => $error, 'message' => 'Impossible de calculer la distance routière.']);
        curl_close($ch);
        exit;
    }
    
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    writeLog("Code HTTP reçu: $httpCode");
    curl_close($ch);
    
    // Vérifier si la réponse est valide
    if ($httpCode != 200) {
        writeLog("ERREUR: API Error: HTTP Code $httpCode");
        writeLog("Réponse: $response");
        
        // Retourner une erreur indiquant qu'aucune route n'a été trouvée
        echo json_encode([
            'error' => 'API Error',
            'message' => 'Erreur lors de la communication avec l\'API Google Maps.',
            'http_code' => $httpCode
        ]);
        exit;
    }
    
    // Analyser la réponse JSON
    $responseData = json_decode($response, true);
    writeLog("Réponse reçue: " . substr(json_encode($responseData), 0, 500) . "...");
    
    // Vérifier le statut de la réponse
    if ($responseData['status'] !== 'OK') {
        writeLog("ERREUR: Google Maps API status: " . $responseData['status']);
        echo json_encode([
            'error' => 'API Error',
            'message' => 'Google Maps API a retourné une erreur: ' . $responseData['status'],
            'status' => $responseData['status']
        ]);
        exit;
    }
    
    // Vérifier si nous avons des résultats
    if (empty($responseData['rows']) || 
        empty($responseData['rows'][0]['elements']) || 
        $responseData['rows'][0]['elements'][0]['status'] !== 'OK') {
        
        $elementStatus = isset($responseData['rows'][0]['elements'][0]['status']) 
            ? $responseData['rows'][0]['elements'][0]['status'] 
            : 'UNKNOWN_ERROR';
        
        writeLog("ERREUR: Pas de route trouvée. Status: $elementStatus");
        echo json_encode([
            'error' => 'No route found',
            'message' => 'Impossible de trouver une route entre ces points.',
            'status' => $elementStatus
        ]);
        exit;
    }
    
    // Extraire la distance et la durée
    $distance = $responseData['rows'][0]['elements'][0]['distance']['value']; // en mètres
    $duration = $responseData['rows'][0]['elements'][0]['duration']['value']; // en secondes
    $distanceText = $responseData['rows'][0]['elements'][0]['distance']['text']; // formaté (ex: "5,2 km")
    $durationText = $responseData['rows'][0]['elements'][0]['duration']['text']; // formaté (ex: "10 minutes")
    
    writeLog("Distance extraite: $distance mètres ($distanceText)");
    writeLog("Durée extraite: $duration secondes ($durationText)");
    
    // Formater la réponse pour être compatible avec le format attendu par le code JavaScript
    // Nous créons une structure similaire à celle d'OpenRouteService pour minimiser les changements côté client
    $formattedResponse = [
        'type' => 'FeatureCollection',
        'features' => [
            [
                'properties' => [
                    'summary' => [
                        'distance' => $distance,
                        'duration' => $duration
                    ],
                    'text_distance' => $distanceText,
                    'text_duration' => $durationText
                ]
            ]
        ]
    ];
    
    // Renvoyer la réponse formatée
    echo json_encode($formattedResponse);
    writeLog("Réponse formatée envoyée au client avec succès");
    
} catch (Exception $e) {
    $error = 'Exception: ' . $e->getMessage();
    writeLog("ERREUR: $error");
    echo json_encode(['error' => $error, 'message' => 'Une erreur est survenue lors du calcul de la distance.']);
    exit;
}
