<?php
// Fichier : test-distance.php
// But : tester directement le calcul de distance avec des coordonnées fixes

// Définir que la réponse sera en HTML
header('Content-Type: text/html; charset=utf-8');

// Activer l'affichage des erreurs pour le débogage
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

echo "<h1>Test de calcul de distance avec Google Maps API</h1>";

// Coordonnées de test (La Réunion)
$startLng = 55.536384;  // Longitude
$startLat = -21.115141; // Latitude
$endLng = 55.450000;    // Longitude
$endLat = -21.200000;   // Latitude

echo "<p>Coordonnées de départ: [$startLat, $startLng]</p>";
echo "<p>Coordonnées d'arrivée: [$endLat, $endLng]</p>";

// Tester l'API Google Maps
echo "<h2>Test de l'API Google Maps</h2>";

// Construire l'URL pour notre API - Utiliser le chemin complet
// Déterminer le chemin du script actuel
$scriptPath = dirname($_SERVER['SCRIPT_NAME']);
$apiPath = $scriptPath . '/get-distance.php';

// Construire l'URL complète
$apiUrl = $apiPath . "?start=$startLng,$startLat&end=$endLng,$endLat";
echo "<p>URL de l'API: " . htmlspecialchars($apiUrl) . "</p>";

// Appeler notre API
$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $apiUrl);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
curl_setopt($ch, CURLOPT_TIMEOUT, 20);

$response = curl_exec($ch);

// Gère les erreurs cURL
if (curl_errno($ch)) {
    echo "<h3>Erreur cURL</h3>";
    echo "<p>Erreur: " . htmlspecialchars(curl_error($ch)) . "</p>";
    echo "<p>Essayons une approche alternative...</p>";
    
    // Approche alternative: utiliser file_get_contents au lieu de cURL
    echo "<h3>Test avec file_get_contents</h3>";
    
    // Construire l'URL avec le nom d'hôte complet
    $host = $_SERVER['HTTP_HOST'];
    $fullApiUrl = "http://$host$apiPath?start=$startLng,$startLat&end=$endLng,$endLat";
    echo "<p>URL complète: " . htmlspecialchars($fullApiUrl) . "</p>";
    
    // Désactiver les avertissements pour file_get_contents
    $context = stream_context_create([
        'http' => [
            'ignore_errors' => true,
            'timeout' => 10
        ]
    ]);
    
    $altResponse = @file_get_contents($fullApiUrl, false, $context);
    
    if ($altResponse === false) {
        echo "<p>Erreur avec file_get_contents également.</p>";
        
        // Dernière tentative: appel direct à la fonction
        echo "<h3>Test direct de la fonction</h3>";
        echo "<p>Appel direct à la fonction de calcul de distance...</p>";
        
        // Inclure le fichier get-distance.php mais capturer sa sortie
        ob_start();
        $_GET['start'] = "$startLng,$startLat";
        $_GET['end'] = "$endLng,$endLat";
        include 'get-distance.php';
        $directOutput = ob_get_clean();
        
        echo "<h4>Résultat de l'appel direct:</h4>";
        echo "<pre>" . htmlspecialchars($directOutput) . "</pre>";
        
        // Essayer de parser la sortie directe
        try {
            $directData = json_decode($directOutput, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                echo "<h4>Données parsées de l'appel direct:</h4>";
                echo "<pre>" . htmlspecialchars(json_encode($directData, JSON_PRETTY_PRINT)) . "</pre>";
                
                // Extraire la distance si disponible
                if (isset($directData['features'][0]['properties']['summary']['distance'])) {
                    $distance = $directData['features'][0]['properties']['summary']['distance'];
                    $distanceKm = $distance / 1000;
                    echo "<h3>Distance routière calculée (appel direct)</h3>";
                    echo "<p><strong>$distanceKm km</strong> ($distance mètres)</p>";
                    
                    if (isset($directData['features'][0]['properties']['text_distance'])) {
                        echo "<p>Distance formatée: " . htmlspecialchars($directData['features'][0]['properties']['text_distance']) . "</p>";
                    }
                    
                    if (isset($directData['features'][0]['properties']['summary']['duration'])) {
                        $duration = $directData['features'][0]['properties']['summary']['duration'];
                        $durationMin = round($duration / 60);
                        echo "<p>Durée: <strong>$durationMin minutes</strong> ($duration secondes)</p>";
                        
                        if (isset($directData['features'][0]['properties']['text_duration'])) {
                            echo "<p>Durée formatée: " . htmlspecialchars($directData['features'][0]['properties']['text_duration']) . "</p>";
                        }
                    }
                } else if (isset($directData['error'])) {
                    echo "<h3>Erreur retournée par l'API</h3>";
                    echo "<p>" . htmlspecialchars($directData['error']) . "</p>";
                    if (isset($directData['message'])) {
                        echo "<p>" . htmlspecialchars($directData['message']) . "</p>";
                    }
                    if (isset($directData['status'])) {
                        echo "<p>Status: " . htmlspecialchars($directData['status']) . "</p>";
                    }
                }
            } else {
                echo "<p>Erreur de parsing JSON (appel direct): " . json_last_error_msg() . "</p>";
            }
        } catch (Exception $e) {
            echo "<p>Exception lors du parsing (appel direct): " . htmlspecialchars($e->getMessage()) . "</p>";
        }
    } else {
        echo "<h4>Réponse avec file_get_contents:</h4>";
        echo "<pre>" . htmlspecialchars($altResponse) . "</pre>";
        
        // Essayer de parser la réponse
        try {
            $altData = json_decode($altResponse, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                echo "<h4>Données parsées:</h4>";
                echo "<pre>" . htmlspecialchars(json_encode($altData, JSON_PRETTY_PRINT)) . "</pre>";
                
                // Extraire la distance si disponible
                if (isset($altData['features'][0]['properties']['summary']['distance'])) {
                    $distance = $altData['features'][0]['properties']['summary']['distance'];
                    $distanceKm = $distance / 1000;
                    echo "<h3>Distance routière calculée (file_get_contents)</h3>";
                    echo "<p><strong>$distanceKm km</strong> ($distance mètres)</p>";
                    
                    if (isset($altData['features'][0]['properties']['text_distance'])) {
                        echo "<p>Distance formatée: " . htmlspecialchars($altData['features'][0]['properties']['text_distance']) . "</p>";
                    }
                    
                    if (isset($altData['features'][0]['properties']['summary']['duration'])) {
                        $duration = $altData['features'][0]['properties']['summary']['duration'];
                        $durationMin = round($duration / 60);
                        echo "<p>Durée: <strong>$durationMin minutes</strong> ($duration secondes)</p>";
                        
                        if (isset($altData['features'][0]['properties']['text_duration'])) {
                            echo "<p>Durée formatée: " . htmlspecialchars($altData['features'][0]['properties']['text_duration']) . "</p>";
                        }
                    }
                } else if (isset($altData['error'])) {
                    echo "<h3>Erreur retournée par l'API</h3>";
                    echo "<p>" . htmlspecialchars($altData['error']) . "</p>";
                    if (isset($altData['message'])) {
                        echo "<p>" . htmlspecialchars($altData['message']) . "</p>";
                    }
                    if (isset($altData['status'])) {
                        echo "<p>Status: " . htmlspecialchars($altData['status']) . "</p>";
                    }
                }
            } else {
                echo "<p>Erreur de parsing JSON: " . json_last_error_msg() . "</p>";
            }
        } catch (Exception $e) {
            echo "<p>Exception lors du parsing: " . htmlspecialchars($e->getMessage()) . "</p>";
        }
    }
    
    curl_close($ch);
    exit;
}

$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
echo "<p>Code HTTP reçu: $httpCode</p>";
curl_close($ch);

// Afficher la réponse brute
echo "<h3>Réponse brute</h3>";
echo "<pre>" . htmlspecialchars($response) . "</pre>";

// Essayer de parser la réponse JSON
try {
    $data = json_decode($response, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        echo "<h3>Erreur de parsing JSON</h3>";
        echo "<p>Erreur: " . json_last_error_msg() . "</p>";
    } else {
        echo "<h3>Réponse parsée</h3>";
        echo "<pre>" . htmlspecialchars(json_encode($data, JSON_PRETTY_PRINT)) . "</pre>";
        
        // Extraire et afficher la distance
        if (isset($data['features'][0]['properties']['summary']['distance'])) {
            $distance = $data['features'][0]['properties']['summary']['distance'];
            $distanceKm = $distance / 1000;
            echo "<h3>Distance routière calculée</h3>";
            echo "<p><strong>$distanceKm km</strong> ($distance mètres)</p>";
            
            if (isset($data['features'][0]['properties']['text_distance'])) {
                echo "<p>Distance formatée: " . htmlspecialchars($data['features'][0]['properties']['text_distance']) . "</p>";
            }
            
            if (isset($data['features'][0]['properties']['summary']['duration'])) {
                $duration = $data['features'][0]['properties']['summary']['duration'];
                $durationMin = round($duration / 60);
                echo "<p>Durée: <strong>$durationMin minutes</strong> ($duration secondes)</p>";
                
                if (isset($data['features'][0]['properties']['text_duration'])) {
                    echo "<p>Durée formatée: " . htmlspecialchars($data['features'][0]['properties']['text_duration']) . "</p>";
                }
            }
        } else if (isset($data['error'])) {
            echo "<h3>Erreur retournée par l'API</h3>";
            echo "<p>" . htmlspecialchars($data['error']) . "</p>";
            if (isset($data['message'])) {
                echo "<p>" . htmlspecialchars($data['message']) . "</p>";
            }
            if (isset($data['status'])) {
                echo "<p>Status: " . htmlspecialchars($data['status']) . "</p>";
            }
        } else {
            echo "<h3>Impossible de trouver la distance dans la réponse</h3>";
        }
    }
} catch (Exception $e) {
    echo "<h3>Exception</h3>";
    echo "<p>Erreur: " . htmlspecialchars($e->getMessage()) . "</p>";
}
