// Script pour trouver et corriger l'erreur de syntaxe avec ""directe""
const fs = require('fs');
const path = require('path');

// Fonction pour parcourir récursivement un répertoire
function walkDir(dir, callback) {
  fs.readdirSync(dir).forEach(f => {
    let dirPath = path.join(dir, f);
    let isDirectory = fs.statSync(dirPath).isDirectory();
    isDirectory ? walkDir(dirPath, callback) : callback(path.join(dir, f));
  });
}

// Fonction pour rechercher des motifs problématiques dans un fichier
function findSyntaxIssues(filePath) {
  // Vérifier si c'est un fichier JavaScript
  if (!filePath.endsWith('.js')) return;
  
  try {
    const content = fs.readFileSync(filePath, 'utf8');
    
    // Rechercher des motifs problématiques qui pourraient causer l'erreur
    // Par exemple: "distance directe" mal formaté dans une fonction ou condition
    const problematicPatterns = [
      /\("directe"\s/g,       // ""directe"" suivi d'un espace après une parenthèse ouvrante
      /\s+"directe"\)/g,      // ""directe"" suivi d'une parenthèse fermante
      /\("distance directe"/g, // "distance directe" après une parenthèse ouvrante sans guillemets
      /if\s*\(\s*"directe"/g, // ""directe"" utilisé comme condition sans être une variable
      /function\s*\(\s*"directe"/g, // ""directe"" utilisé comme paramètre de fonction sans virgule
      /console\.log\("directe"/g,   // ""directe"" utilisé comme argument de console.log sans guillemets
      /\bdirecte\b(?!\s*=|\s*\+|\s*-|\s*\*|\s*\/|\s*\?|\s*:|\s*;|\s*,|\s*\)|\s*\]|\s*\})/g // ""directe"" utilisé comme identifiant sans être une variable
    ];
    
    let found = false;
    problematicPatterns.forEach((pattern, index) => {
      if (pattern.test(content)) {
        console.log(`⚠️ Motif problématique #${index+1} trouvé dans: ${filePath}`);
        found = true;
        
        // Afficher les lignes contenant le motif
        const lines = content.split('\n');
        lines.forEach((line, lineNum) => {
          pattern.lastIndex = 0; // Réinitialiser le regex pour chaque ligne
          if (pattern.test(line)) {
            console.log(`Ligne ${lineNum+1}: ${line}`);
          }
        });
      }
    });
    
    // Rechercher spécifiquement les cas où "distance directe" n'est pas entre guillemets
    const lines = content.split('\n');
    lines.forEach((line, lineNum) => {
      if (line.includes('distance "directe"') && 
          !line.includes('"distance directe"') && 
          !line.includes("'distance "directe"'") &&
          !line.includes('`distance "directe"`') &&
          !line.includes('// ')) {
        console.log(`⚠️ "distance directe" sans guillemets à la ligne ${lineNum+1} dans: ${filePath}`);
        console.log(`Ligne ${lineNum+1}: ${line}`);
        found = true;
      }
    });
    
    if (found) {
      return filePath;
    }
  } catch (err) {
    console.error(`Erreur lors de la lecture de ${filePath}: ${err}`);
  }
  
  return null;
}

// Fonction pour corriger automatiquement les problèmes courants
function fixSyntaxIssues(filePath) {
  if (!filePath) return;
  
  try {
    let content = fs.readFileSync(filePath, 'utf8');
    let originalContent = content;
    
    // Corrections automatiques pour les cas courants
    const fixes = [
      // Mettre "distance directe" entre guillemets quand il n'est pas dans un commentaire
      { 
        pattern: /([^'"`:])"distance directe"([^'"`])/g, 
        replacement: '$1"distance directe"$2' 
      },
      // Corriger ""directe"" utilisé comme identifiant sans être une variable
      { 
        pattern: /\bdirecte\b(?!\s*=|\s*\+|\s*-|\s*\*|\s*\/|\s*\?|\s*:|\s*;|\s*,|\s*\)|\s*\]|\s*\})/g, 
        replacement: '""directe""' 
      },
      // Corriger les cas spécifiques comme console.log("directe")
      { 
        pattern: /console\.log\("directe"/g, 
        replacement: 'console.log(""directe""' 
      }
    ];
    
    // Appliquer les corrections
    fixes.forEach(fix => {
      content = content.replace(fix.pattern, fix.replacement);
    });
    
    // Vérifier si des modifications ont été apportées
    if (content !== originalContent) {
      console.log(`✅ Corrections appliquées à: ${filePath}`);
      fs.writeFileSync(filePath, content, 'utf8');
    }
  } catch (err) {
    console.error(`Erreur lors de la correction de ${filePath}: ${err}`);
  }
}

// Répertoire de départ
const startDir = '/home/project';

// Rechercher les problèmes
console.log("Recherche des problèmes de syntaxe avec '"directe"'...");
const problematicFiles = [];
walkDir(startDir, (filePath) => {
  const result = findSyntaxIssues(filePath);
  if (result) {
    problematicFiles.push(result);
  }
});

// Corriger les problèmes
if (problematicFiles.length > 0) {
  console.log("\nApplication des corrections aux fichiers problématiques...");
  problematicFiles.forEach(filePath => {
    fixSyntaxIssues(filePath);
  });
} else {
  console.log("Aucun problème évident trouvé. Vérification des fichiers principaux...");
  
  // Vérifier et corriger les fichiers principaux même si aucun problème n'a été détecté
  const mainFiles = [
    '/home/project/js/geoloc.js',
    '/home/project/tmpl/assets/main.js'
  ];
  
  mainFiles.forEach(filePath => {
    if (fs.existsSync(filePath)) {
      console.log(`Vérification du fichier principal: ${filePath}`);
      fixSyntaxIssues(filePath);
    }
  });
}

console.log("\nVérification terminée. Exécutez le script de vérification de syntaxe pour confirmer les corrections.");
