document.addEventListener('DOMContentLoaded', () => {
    console.log("JS chargé (geoloc.js)");

    const geolocateButton = document.querySelector('.geolocation-button');
    const farmerList = document.querySelector('.farmer-list');
    const distancesInfo = document.querySelector('.distances-info');

    if (!geolocateButton || !farmerList) {
        console.error("Éléments DOM manquants");
        return;
    }

    geolocateButton.addEventListener('click', () => {
        if (!navigator.geolocation) {
            alert("Géolocalisation non supportée par votre navigateur.");
            return;
        }

        geolocateButton.disabled = true;
        geolocateButton.innerHTML = '⏳ Localisation...';

        navigator.geolocation.getCurrentPosition(async (position) => {
            const userLat = position.coords.latitude;
            const userLng = position.coords.longitude;
            console.log("Position utilisateur:", userLat, userLng);

            if (distancesInfo) {
                distancesInfo.style.display = 'block';
            }

            const items = document.querySelectorAll('.farmer-item');
            const distances = [];
            let successCount = 0;
            let errorCount = 0;
            let totalRequests = 0;
            let completedRequests = 0;

            // Fonction pour mettre à jour le bouton avec le progrès
            const updateProgress = () => {
                const percent = totalRequests > 0 ? Math.round((completedRequests / totalRequests) * 100) : 0;
                geolocateButton.innerHTML = `⏳ Calcul des distances... ${percent}%`;
            };

            // Traiter chaque agriculteur
            for (const item of items) {
                const id = item.getAttribute('data-id');
                const coords = item.getAttribute('data-coordinates');
                
                if (!coords) {
                    console.warn(`Agriculteur ID ${id}: Pas de coordonnées disponibles`);
                    continue;
                }
                
                const [farmLat, farmLng] = coords.split(',').map(parseFloat);
                
                // Vérifier que les coordonnées sont valides
                if (isNaN(farmLat) || isNaN(farmLng)) {
                    console.warn(`Coordonnées invalides pour l'agriculteur ID ${id}: ${coords}`);
                    continue;
                }
                
                totalRequests++;
                updateProgress();
                
                console.log(`Calcul de distance pour agriculteur ${id}: [${farmLat}, ${farmLng}]`);
                
                try {
                    // Essayer d'obtenir la distance routière via l'API Google Maps
                    // IMPORTANT: Notre API attend toujours le format "longitude,latitude"
                    // mais en interne elle convertira pour Google Maps qui attend "latitude,longitude"
                    
                    // Construire l'URL complète avec le chemin du module
                    const moduleBasePath = '/modules/mod_agriculteur_liste';
                    const url = `${moduleBasePath}/api/get-distance.php?start=${userLng},${userLat}&end=${farmLng},${farmLat}`;
                    console.log(`Appel API pour ID ${id}:`, url);
                    
                    const response = await fetch(url);
                    
                    // Ajouter un log pour voir le statut de la réponse
                    console.log(`Statut de la réponse pour ID ${id}: ${response.status}`);
                    
                    const responseText = await response.text();
                    
                    console.log(`Réponse brute pour ID ${id}:`, responseText);
                    
                    // Essayer de parser la réponse JSON
                    let data;
                    try {
                        data = JSON.parse(responseText);
                    } catch (parseError) {
                        console.error(`Erreur de parsing JSON pour ID ${id}:`, parseError);
                        throw new Error(`Réponse non-JSON: ${responseText.substring(0, 100)}...`);
                    }
                    
                    console.log(`Réponse API pour ID ${id}:`, data);
                    
                    // Vérifier si la réponse contient une erreur
                    if (data.error) {
                        throw new Error(`Erreur API: ${data.error}`);
                    }
                    
                    // Extraire la distance de la réponse
                    if (data.features && 
                        data.features[0] && 
                        data.features[0].properties && 
                        data.features[0].properties.summary && 
                        typeof data.features[0].properties.summary.distance === 'number') {
                        
                        // Convertir de mètres en kilomètres
                        const distanceKm = data.features[0].properties.summary.distance / 1000;
                        console.log(`Distance routière calculée pour ID ${id}: ${distanceKm.toFixed(1)} km`);
                        
                        // Mettre à jour l'affichage de la distance
                        const distEl = document.getElementById('distance-' + id);
                        if (distEl) {
                            const valueEl = distEl.querySelector('.distance-value');
                            if (valueEl) {
                                // Mettre à jour le contenu textuel
                                valueEl.textContent = distanceKm.toFixed(1);
                                distEl.classList.add('road-distance');
                                distEl.style.display = 'block';
                                console.log(`Mise à jour de l'affichage pour ID ${id}: ${distanceKm.toFixed(1)} km`);
                            }
                        }

                        // Mettre à jour aussi dans la vue grille si elle existe
                        const gridDistEl = document.getElementById('grid-distance-' + id);
                        if (gridDistEl) {
                            const gridValueEl = gridDistEl.querySelector('.grid-distance-value');
                            if (gridValueEl) {
                                // Mettre à jour le contenu textuel
                                gridValueEl.textContent = distanceKm.toFixed(1);
                                gridDistEl.classList.add('road-distance');
                            }
                        }

                        // Stocker pour le tri
                        distances.push({ id, distance: distanceKm, element: item });
                        item.setAttribute('data-distance', distanceKm);
                        successCount++;
                    } else {
                        console.warn(`Structure de réponse inattendue pour ID ${id}`);
                        throw new Error('Structure de réponse inattendue');
                    }
                } catch (error) {
                    console.error(`Erreur lors du calcul de la distance pour ID ${id}:`, error);
                    errorCount++;
                    
                    // Masquer l'élément de distance en cas d'erreur
                    const distEl = document.getElementById('distance-' + id);
                    if (distEl) {
                        distEl.style.display = 'none';
                    }
                    
                    // Masquer aussi dans la vue grille
                    const gridDistEl = document.getElementById('grid-distance-' + id);
                    if (gridDistEl) {
                        gridDistEl.style.display = 'none';
                    }
                } finally {
                    completedRequests++;
                    updateProgress();
                }
            }

            console.log(`Statistiques: ${successCount} distances routières calculées, ${errorCount} erreurs`);

            // Trier les éléments par distance (uniquement ceux avec des distances calculées)
            if (distances.length > 0) {
                distances.sort((a, b) => a.distance - b.distance);
                
                // Réorganiser les éléments dans le DOM
                distances.forEach(d => {
                    const parent = d.element.parentNode;
                    if (parent) {
                        farmerList.appendChild(parent);
                    }
                });
            }

            // Réinitialiser le bouton
            geolocateButton.disabled = false;
            geolocateButton.innerHTML = '📍 Trouver les agriculteurs proches de moi';
            
            // Afficher un message approprié
            if (successCount > 0) {
                // Afficher un message de succès si au moins une distance a été calculée
                const successMessage = document.createElement('div');
                successMessage.className = 'geolocation-success';
                successMessage.innerHTML = `<i class="fas fa-check-circle"></i> ${successCount} distances calculées avec succès !`;
                document.body.appendChild(successMessage);
                setTimeout(() => successMessage.remove(), 3000);
            } else {
                // Afficher un message d'erreur si aucune distance n'a pu être calculée
                const errorMessage = document.createElement('div');
                errorMessage.className = 'geolocation-error';
                errorMessage.innerHTML = '<i class="fas fa-exclamation-circle"></i> Impossible de calculer les distances routières.';
                document.body.appendChild(errorMessage);
                setTimeout(() => errorMessage.remove(), 3000);
            }
            
            console.log("Calcul des distances terminé");
        }, (error) => {
            console.error("Erreur de géolocalisation:", error);
            alert("Impossible d'obtenir votre position.");
            geolocateButton.disabled = false;
            geolocateButton.innerHTML = '📍 Trouver les agriculteurs proches de moi';
        }, {
            enableHighAccuracy: true,
            timeout: 10000,
            maximumAge: 0
        });
    });
});
