<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_liste
 *
 * @copyright   Copyright (C) 2025 Votre Nom. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;

// Récupérer la base URL du site
$root = Uri::root();
$app = Factory::getApplication();

// Paramètres de pagination
$itemsPerPage = (int)$params->get('items_per_page', 5); // Nombre d'agriculteurs par page
$currentPage = max(1, $app->input->getInt('page', 1)); // Page actuelle
$startIndex = ($currentPage - 1) * $itemsPerPage;
$totalPages = ceil(count($agriculteurs) / $itemsPerPage);

// Détecter la vue active (liste ou grille) à partir des paramètres d'URL
$activeView = $app->input->getCmd('view_mode', 'list'); // Par défaut: liste

// Récupérer la catégorie à filtrer depuis l'URL
$filterCategory = $app->input->getString('category', '');

// Sous-ensemble d'agriculteurs pour la page actuelle
$pageAgriculteurs = array_slice($agriculteurs, $startIndex, $itemsPerPage);

// Construire l'URL pour la pagination avec SP Page Builder et préserver la vue active et la catégorie
$queryParams = [
    'option' => 'com_sppagebuilder', 
    'view' => 'page', 
    'id' => $params->get('sppb_page_id', '50'), 
    'view_mode' => $activeView
];

// Ajouter le paramètre de catégorie s'il existe
if (!empty($filterCategory)) {
    $queryParams['category'] = $filterCategory;
}

$baseUrl = $root . 'index.php?' . http_build_query(array_merge(
    $queryParams,
    array_diff_key($_GET, array_flip(['option', 'view', 'id', 'view_mode', 'category', 'page']))
));

// URL spécifique à votre site pour les fiches agriculteurs
$profileUrlBase = $root . 'index.php/nos-agriculteurs';

// Inclure le fichier CSS externe
$document = Factory::getDocument();
$document->addStyleSheet($root . 'modules/mod_agriculteur_liste/tmpl/assets/styles.css');
?>

<div class="agriculteur-liste-container">
    <!-- Popup de recherche -->
    <div id="search-popup" class="search-popup">
        <div class="search-popup-content">
            <h2>Rechercher</h2>
            <div class="search-form">
                <input type="text" id="search-input" placeholder="Entrez un ou plusieurs mots..." autocomplete="off">
                <button id="search-submit">
                    <i class="fas fa-search"></i>
                </button>
            </div>
            <button id="search-close" class="search-close-button">
                <i class="fas fa-times"></i>
            </button>
        </div>
    </div>

    <!-- Barre de navigation supérieure avec boutons -->
    <div class="top-navigation-bar">
        <a href="index.php?option=com_sppagebuilder&view=page&id=52" class="nav-button back-button">
            <i class="fas fa-arrow-left"></i>
        </a>
        <div class="view-buttons">
            <button id="map-view-button" class="nav-button">
                <i class="fas fa-map-marked-alt"></i>
            </button>
            <button id="grid-view-button" class="nav-button <?php echo ($activeView == 'grid') ? 'active' : ''; ?>">
                <i class="fas fa-th"></i>
            </button>
            <button id="search-button" class="nav-button">
                <i class="fas fa-search"></i>
            </button>
        </div>
    </div>
    
    <!-- Conteneur pour la vue carte (initialement caché) -->
    <div id="map-container" class="map-container" style="display: none;">
        <div id="google-map" class="google-map"></div>
    </div>
    
    <!-- Conteneur pour la vue grille (affichée par défaut) -->
    <div id="grid-container" class="grid-container">
        <!-- Bouton de géolocalisation -->
        <button id="geolocate-button-<?php echo $module->id; ?>" class="geolocation-button">
            <i class="fas fa-location-crosshairs"></i> Les plus proches
        </button>
        
        <div id="distances-info-<?php echo $module->id; ?>" class="distances-info" style="display: none;">
            <i class="fas fa-info-circle"></i> 
            Distances calculées depuis votre position actuelle.
        </div>
        
        <!-- Vue liste -->
        <div id="list-view" class="view-container <?php echo ($activeView == 'list') ? 'active-view' : ''; ?>">
            <ul class="farmer-list" id="farmer-list-<?php echo $module->id; ?>">
                <?php foreach ($pageAgriculteurs as $agriculteur): ?>
                <li>
                    <?php 
                    // Construction du lien vers le profil avec l'URL exacte de votre site
                    // Ajout des paramètres page_retour, view_mode et category pour conserver le contexte
                    $profileUrl = $profileUrlBase . '?submission_id=' . $agriculteur['id'] 
                        . '&page_retour=' . $currentPage 
                        . '&view_mode=' . $activeView;
                    
                    // Ajouter le paramètre de catégorie s'il existe
                    if (!empty($filterCategory)) {
                        $profileUrl .= '&category_retour=' . urlencode($filterCategory);
                    }
                    
                    // Vérifier si l'image existe
                    $imageUrl = !empty($agriculteur['image']) ? $agriculteur['image'] : $root . 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                    ?>
                    <a href="<?php echo $profileUrl; ?>" class="farmer-item" 
                       data-coordinates="<?php echo htmlspecialchars($agriculteur['coordinates']); ?>"
                       data-id="<?php echo $agriculteur['id']; ?>"
                       data-image="<?php echo $imageUrl; ?>"
                       data-name="<?php echo htmlspecialchars($agriculteur['societe']); ?>"
                       data-owner="<?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?>"
                       <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                       data-categories="<?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?>"
                       <?php endif; ?>
                       <?php if (isset($agriculteur['description'])): ?>
                       data-description="<?php echo htmlspecialchars($agriculteur['description']); ?>"
                       <?php endif; ?>
                       <?php if (isset($agriculteur['location'])): ?>
                       data-location="<?php echo htmlspecialchars($agriculteur['location']); ?>"
                       <?php endif; ?>
                      >
                        <div class="farmer-content">
                            <img src="<?php echo $imageUrl; ?>" alt="<?php echo htmlspecialchars($agriculteur['societe']); ?>" class="farmer-image" onerror="this.src='<?php echo $root; ?>media/mod_agriculteur_liste/images/default-farmer.jpg'">
                            <div class="farmer-info">
                                <h3 class="farmer-name"><?php echo htmlspecialchars($agriculteur['societe']); ?></h3>
                                <p class="farmer-owner"><?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?></p>
                                <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                                <p class="farmer-categories"><?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?></p>
                                <?php endif; ?>
                            </div>
                            <div class="farmer-distance" id="distance-<?php echo $agriculteur['id']; ?>">
                                <p class="distance-value"><?php echo number_format($agriculteur['distance'], 1, '.', ''); ?></p>
                                <p class="distance-unit">km</p>
                            </div>
                        </div>
                    </a>
                </li>
                <?php endforeach; ?>
            </ul>
        </div>
        
        <!-- Vue grille (masquée, sera affichée en JS) -->
        <div id="grid-view" class="view-container <?php echo ($activeView == 'grid') ? 'active-view' : ''; ?>">
            <div class="farmer-grid" id="farmer-grid-<?php echo $module->id; ?>">
                <?php foreach ($pageAgriculteurs as $agriculteur): ?>
                <?php 
                // Construction du lien vers le profil avec l'URL exacte de votre site
                // Ajout des paramètres page_retour, view_mode et category pour conserver le contexte
                $profileUrl = $profileUrlBase . '?submission_id=' . $agriculteur['id'] 
                    . '&page_retour=' . $currentPage 
                    . '&view_mode=' . $activeView;
                
                // Ajouter le paramètre de catégorie s'il existe
                if (!empty($filterCategory)) {
                    $profileUrl .= '&category_retour=' . urlencode($filterCategory);
                }
                
                // Vérifier si l'image existe
                $imageUrl = !empty($agriculteur['image']) ? $agriculteur['image'] : $root . 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                ?>
                <a href="<?php echo $profileUrl; ?>" class="grid-item" 
                   data-coordinates="<?php echo htmlspecialchars($agriculteur['coordinates']); ?>"
                   data-id="<?php echo $agriculteur['id']; ?>"
                   data-image="<?php echo $imageUrl; ?>"
                   data-name="<?php echo htmlspecialchars($agriculteur['societe']); ?>"
                   data-owner="<?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?>"
                   <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                   data-categories="<?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?>"
                   <?php endif; ?>
                   <?php if (isset($agriculteur['description'])): ?>
                   data-description="<?php echo htmlspecialchars($agriculteur['description']); ?>"
                   <?php endif; ?>
                   <?php if (isset($agriculteur['location'])): ?>
                   data-location="<?php echo htmlspecialchars($agriculteur['location']); ?>"
                   <?php endif; ?>
                  >
                    <div class="grid-item-content">
                        <div class="grid-item-image-container">
                            <img src="<?php echo $imageUrl; ?>" alt="<?php echo htmlspecialchars($agriculteur['societe']); ?>" class="grid-item-image" onerror="this.src='<?php echo $root; ?>media/mod_agriculteur_liste/images/default-farmer.jpg'">
                            <div class="grid-item-distance" id="grid-distance-<?php echo $agriculteur['id']; ?>">
                                <span class="grid-distance-value"><?php echo number_format($agriculteur['distance'], 1, '.', ''); ?></span>
                                <span class="grid-distance-unit">km</span>
                            </div>
                        </div>
                        <div class="grid-item-info">
                            <h3 class="grid-item-name"><?php echo htmlspecialchars($agriculteur['societe']); ?></h3>
                            <p class="grid-item-owner"><?php echo htmlspecialchars($agriculteur['prenom'] . ' ' . $agriculteur['nom']); ?></p>
                            <?php if (isset($agriculteur['categories']) && is_array($agriculteur['categories'])): ?>
                            <p class="grid-item-categories"><?php echo htmlspecialchars(implode(', ', $agriculteur['categories'])); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                </a>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- Conteneur des résultats de recherche -->
        <div id="search-results-container" class="view-container" style="display: none;">
            <h3 class="search-results-title">Résultats de recherche</h3>
            <p id="search-count" class="search-count"></p>
            <ul class="farmer-list" id="search-results-list">
                <!-- Les résultats de recherche seront injectés ici par JavaScript -->
            </ul>
        </div>
        
        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
        <div class="pagination-container">
            <div class="pagination">
                <!-- Retour à la première page -->
                <?php if ($currentPage > 1): ?>
                <a href="<?php echo $baseUrl . '&page=1'; ?>" class="page-item first">
                    <i class="fas fa-angle-double-left"></i>
                </a>
                <?php endif; ?>
                
                <!-- Page précédente -->
                <?php if ($currentPage > 1): ?>
                <a href="<?php echo $baseUrl . '&page=' . ($currentPage - 1); ?>" class="page-item prev">
                    <i class="fas fa-chevron-left"></i>
                </a>
                <?php endif; ?>
                
                <?php
                // Afficher les numéros de page avec limitation
                $maxPagesToShow = 5;
                $startPage = max(1, min($currentPage - floor($maxPagesToShow / 2), $totalPages - $maxPagesToShow + 1));
                $endPage = min($totalPages, $startPage + $maxPagesToShow - 1);
                
                // Ajuster startPage si nécessaire
                $startPage = max(1, $endPage - $maxPagesToShow + 1);
                
                // Afficher la première page si nécessaire
                if ($startPage > 1): ?>
                    <a href="<?php echo $baseUrl . '&page=1'; ?>" class="page-item">1</a>
                    <?php if ($startPage > 2): ?>
                    <span class="page-item ellipsis">...</span>
                    <?php endif; ?>
                <?php endif; ?>
                
                <?php for ($i = $startPage; $i <= $endPage; $i++): ?>
                    <?php if ($i == $currentPage): ?>
                    <span class="page-item active"><?php echo $i; ?></span>
                    <?php else: ?>
                    <a href="<?php echo $baseUrl . '&page=' . $i; ?>" class="page-item"><?php echo $i; ?></a>
                    <?php endif; ?>
                <?php endfor; ?>
                
                <?php 
                // Afficher la dernière page si nécessaire
                if ($endPage < $totalPages): ?>
                    <?php if ($endPage < $totalPages - 1): ?>
                    <span class="page-item ellipsis">...</span>
                    <?php endif; ?>
                    <a href="<?php echo $baseUrl . '&page=' . $totalPages; ?>" class="page-item"><?php echo $totalPages; ?></a>
                <?php endif; ?>
                
                <!-- Page suivante -->
                <?php if ($currentPage < $totalPages): ?>
                <a href="<?php echo $baseUrl . '&page=' . ($currentPage + 1); ?>" class="page-item next">
                    <i class="fas fa-chevron-right"></i>
                </a>
                <?php endif; ?>
            </div>
            
            <div class="page-info">
                Page <?php echo $currentPage; ?> sur <?php echo $totalPages; ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
</div>

<!-- Inclusion des scripts JavaScript -->
<script>
// Variables globales partagées entre les scripts
const moduleId = '<?php echo $module->id; ?>';
const root = '<?php echo $root; ?>';
const profileUrlBase = '<?php echo $profileUrlBase; ?>';
const currentPage = <?php echo $currentPage; ?>;
const activeView = '<?php echo $activeView; ?>';
</script>

<!-- Inclure les fichiers JavaScript séparés -->
<script src="<?php echo $root; ?>modules/mod_agriculteur_liste/tmpl/assets/main.js"></script>
