/**
 * Script principal du module agriculteur_liste
 * Ce fichier contient toutes les fonctionnalités essentielles du module
 */

document.addEventListener('DOMContentLoaded', function() {
    // Variables et éléments du DOM
    const geolocateButton = document.getElementById('geolocate-button-' + moduleId);
    const distancesInfo = document.getElementById('distances-info-' + moduleId);
    
    // Références aux boutons de navigation
    const mapViewButton = document.getElementById('map-view-button');
    const gridViewButton = document.getElementById('grid-view-button');
    const searchButton = document.getElementById('search-button');
    
    // Références aux conteneurs de vues
    const mapContainer = document.getElementById('map-container');
    const gridContainer = document.getElementById('grid-container');
    
    // Références aux vues liste et grille
    const listView = document.getElementById('list-view');
    const gridView = document.getElementById('grid-view');
    
    // Références au popup de recherche et aux éléments associés
    const searchPopup = document.getElementById('search-popup');
    const searchInput = document.getElementById('search-input');
    const searchSubmit = document.getElementById('search-submit');
    const searchClose = document.getElementById('search-close');
    const searchResultsContainer = document.getElementById('search-results-container');
    const searchResultsList = document.getElementById('search-results-list');
    const searchCount = document.getElementById('search-count');
    
    // Variables pour le mode recherche
    let isSearchMode = false;
    let currentSearchQuery = '';
    let searchResults = [];
    let searchCurrentPage = 1;
    const searchItemsPerPage = 10;
    
    // Cache pour l'index de recherche
    let searchIndex = null;
    
    // Variables pour la carte Google Maps
    let map;
    let markers = [];
    let markerCluster;
    
    // ====================
    // = FONCTIONS UTILITAIRES
    // ====================
    
    function setViewMode(mode) {
        try {
            localStorage.setItem('agriculteur_view_mode', mode);
        } catch (e) {
            console.error('Impossible de stocker le mode de vue:', e);
        }
        updatePaginationLinks(mode);
    }
    
    function updatePaginationLinks(viewMode) {
        document.querySelectorAll('.pagination a').forEach(function(link) {
            const url = new URL(link.href);
            url.searchParams.set('view_mode', viewMode);
            
            if (isSearchMode && currentSearchQuery) {
                url.searchParams.set('search_query', encodeURIComponent(currentSearchQuery));
            }
            
            link.href = url.toString();
        });
    }
    
    function toRad(degrees) {
        return degrees * Math.PI / 180;
    }
    
    function calculateHaversineDistance(lat1, lng1, lat2, lng2) {
        const R = 6371;
        const dLat = toRad(lat2 - lat1);
        const dLng = toRad(lng2 - lng1);
        
        const a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) *
                Math.sin(dLng / 2) * Math.sin(dLng / 2);
                
        const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        return R * c;
    }
    
    function updateDistanceDisplay(elementId, distance) {
        const distanceElement = document.getElementById(elementId);
        if (!distanceElement) return;
        
        const distanceValue = distanceElement.querySelector('.distance-value') || 
                            distanceElement.querySelector('.grid-distance-value');
        
        if (distanceValue) {
            distanceValue.textContent = distance.toFixed(1);
        }
    }

    function showSuccessMessage() {
        const successMessage = document.createElement('div');
        successMessage.className = 'geolocation-success';
        successMessage.innerHTML = '<i class="fas fa-check-circle"></i> Distances calculées avec succès !';
        
        document.body.appendChild(successMessage);
        
        setTimeout(function() {
            successMessage.remove();
        }, 3000);
    }
    
    // ====================
    // = GESTION DES VUES
    // ====================
    
    const listViewButton = document.createElement('button');
    listViewButton.innerHTML = '<i class="fas fa-list"></i>';
    listViewButton.className = 'nav-button list-view-button';
    listViewButton.style.marginRight = '0px';
    
    if (gridViewButton && gridViewButton.parentNode) {
        gridViewButton.parentNode.insertBefore(listViewButton, gridViewButton);
    }
    
    if (activeView === 'list') {
        listViewButton.classList.add('active');
    }
    
    function checkForSearchQueryInURL() {
        const urlParams = new URLSearchParams(window.location.search);
        const searchQuery = urlParams.get('search_query');
        
        if (searchQuery) {
            const decodedQuery = decodeURIComponent(searchQuery);
            currentSearchQuery = decodedQuery;
            
            if (searchInput) {
                searchInput.value = decodedQuery;
            }
            
            if (!searchIndex) {
                loadSearchIndex();
                setTimeout(() => {
                    performSearch(decodedQuery);
                }, 500);
            } else {
                performSearch(decodedQuery);
            }
            
            return true;
        }
        
        return false;
    }
    
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            isSearchMode = false;
            mapViewButton.classList.add('active');
            gridViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            
            mapContainer.style.display = 'block';
            gridContainer.style.display = 'none';
            
            if (!map) {
                loadGoogleMapsWithClusterer();
            } else {
                google.maps.event.trigger(map, 'resize');
            }
            
            setViewMode('map');
        });
    }
    
    if (gridViewButton) {
        gridViewButton.addEventListener('click', function() {
            isSearchMode = false;
            gridViewButton.classList.add('active');
            mapViewButton.classList.remove('active');
            listViewButton.classList.remove('active');
            
            gridContainer.style.display = 'block';
            mapContainer.style.display = 'none';
            gridView.classList.add('active-view');
            listView.classList.remove('active-view');
            searchResultsContainer.style.display = 'none';
            
            setViewMode('grid');
        });
    }
    
    listViewButton.addEventListener('click', function() {
        isSearchMode = false;
        
        if (mapContainer.style.display !== 'none') {
            mapViewButton.classList.remove('active');
            gridViewButton.classList.remove('active');
            mapContainer.style.display = 'none';
            gridContainer.style.display = 'block';
        }
        
        listView.classList.add('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'none';
        listViewButton.classList.add('active');
        gridViewButton.classList.remove('active');
        
        setViewMode('list');
    });
    
    // ====================
    // = RECHERCHE
    // ====================
    
    if (searchButton) {
        searchButton.addEventListener('click', function() {
            searchPopup.style.display = 'flex';
            searchInput.focus();
            searchInput.value = currentSearchQuery || '';
            
            if (!searchIndex) {
                loadSearchIndex();
            }
        });
    }
    
    if (searchClose) {
        searchClose.addEventListener('click', function() {
            searchPopup.style.display = 'none';
        });
    }
    
    searchPopup.addEventListener('click', function(event) {
        if (event.target === searchPopup) {
            searchPopup.style.display = 'none';
        }
    });
    
    if (searchSubmit) {
        searchSubmit.addEventListener('click', function() {
            const query = searchInput.value.trim();
            if (query) {
                performSearch(query);
            } else {
                alert('Veuillez entrer un terme de recherche');
            }
        });
    }
    
    if (searchInput) {
        searchInput.addEventListener('keypress', function(event) {
            if (event.key === 'Enter') {
                const query = searchInput.value.trim();
                if (query) {
                    performSearch(query);
                } else {
                    alert('Veuillez entrer un terme de recherche');
                }
            }
        });
    }
    
    function loadSearchIndex() {
        fetch('/media/mod_agriculteur_liste/search-index.json')
            .then(response => {
                if (!response.ok) throw new Error(`Erreur HTTP: ${response.status}`);
                return response.json();
            })
            .then(data => {
                if (data && Array.isArray(data.entries)) {
                    searchIndex = data.entries;
                    checkForSearchQueryInURL();
                } else {
                    loadIndexFromDOM();
                }
            })
            .catch(error => {
                console.error('Erreur lors du chargement de l\'index:', error);
                loadIndexFromDOM();
            });
    }
    
    function loadIndexFromDOM() {
        const farmerItems = document.querySelectorAll('.farmer-item, .grid-item');
        searchIndex = [];
        
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const coordinates = item.getAttribute('data-coordinates');
            
            const dataAttributes = {};
            for (let attr of item.attributes) {
                if (attr.name.startsWith('data-') && attr.name !== 'data-id' && 
                    attr.name !== 'data-name' && attr.name !== 'data-owner') {
                    const key = attr.name.replace('data-', '');
                    dataAttributes[key] = attr.value;
                }
            }
            
            const additionalText = Array.from(item.querySelectorAll('p, h3, span, div'))
                .map(el => el.textContent.trim())
                .filter(text => text.length > 0)
                .join(' ');
            
            const searchableText = [
                name || '',
                owner || '',
                additionalText,
                ...Object.values(dataAttributes)
            ].join(' ').toLowerCase();
            
            let image = '';
            const imgElement = item.querySelector('img');
            if (imgElement && imgElement.src) {
                image = imgElement.src;
            }
            
            searchIndex.push({
                id: id,
                name: name || '',
                owner: owner || '',
                coordinates: coordinates || '',
                searchableText: searchableText,
                image: image,
                element: item
            });
        });
        
        checkForSearchQueryInURL();
    }
    
    function performSearch(query) {
        if (!query) query = searchInput.value.trim().toLowerCase();
        else query = query.toLowerCase();
        
        if (!query) {
            alert('Veuillez entrer un terme de recherche');
            return;
        }
        
        if (!searchIndex || searchIndex.length === 0) {
            loadSearchIndex();
            setTimeout(() => performSearch(query), 500);
            return;
        }
        
        isSearchMode = true;
        currentSearchQuery = query;
        searchCurrentPage = 1;
        
        let results = [];
        
        if (searchIndex.length > 0 && searchIndex[0].element) {
            const terms = query.split(/\s+/).filter(term => term.length > 1);
            
            searchIndex.forEach(item => {
                let score = 0;
                const searchableText = item.searchableText || '';
                
                if (searchableText.includes(query)) score += 10;
                
                if (terms.length > 0) {
                    let termMatches = 0;
                    
                    terms.forEach(term => {
                        if (searchableText.includes(term)) {
                            termMatches++;
                            if (item.name.toLowerCase().includes(term)) score += 3;
                            if (item.owner.toLowerCase().includes(term)) score += 2;
                        }
                    });
                    
                    if (termMatches > 0) score += (termMatches / terms.length) * 5;
                }
                
                if (score > 0) results.push({ item: item, score: score });
            });
        } else {
            const terms = query.split(/\s+/).filter(term => term.length > 1);
            
            searchIndex.forEach(item => {
                let score = 0;
                const searchableText = item.searchableText || '';
                
                if (searchableText.includes(query)) score += 10;
                
                if (terms.length > 0) {
                    let termMatches = 0;
                    
                    terms.forEach(term => {
                        if (searchableText.includes(term)) {
                            termMatches++;
                            if (item.name.toLowerCase().includes(term)) score += 3;
                            if (item.owner.toLowerCase().includes(term)) score += 2;
                            if (item.description && item.description.toLowerCase().includes(term)) score += 1;
                            
                            ['categories', 'products', 'labels', 'certifications', 'sales_methods'].forEach(field => {
                                if (Array.isArray(item[field])) {
                                    item[field].forEach(value => {
                                        if (typeof value === 'string' && value.toLowerCase().includes(term)) score += 1;
                                    });
                                }
                            });
                        }
                    });
                    
                    if (termMatches > 0) score += (termMatches / terms.length) * 5;
                }
                
                if (score > 0) results.push({ item: item, score: score });
            });
        }
        
        results.sort((a, b) => b.score - a.score);
        searchResults = results.map(result => result.item);
        searchPopup.style.display = 'none';
        displaySearchResults();
    }
    
    function displaySearchResults() {
        searchResultsList.innerHTML = '';
        
        if (searchResults.length === 0) {
            searchCount.textContent = 'Aucun résultat trouvé pour "' + currentSearchQuery + '"';
            searchResultsList.innerHTML = '<div class="no-results">Aucun résultat ne correspond à votre recherche. Essayez avec un autre terme.</div>';
            hideSearchPagination();
        } else {
            searchCount.textContent = searchResults.length + ' résultat(s) trouvé(s) pour "' + currentSearchQuery + '"';
            
            const startIndex = (searchCurrentPage - 1) * searchItemsPerPage;
            const endIndex = Math.min(startIndex + searchItemsPerPage, searchResults.length);
            const pageResults = searchResults.slice(startIndex, endIndex);
            
            pageResults.forEach(function(result) {
                if (result.element) {
                    const listItem = document.createElement('li');
                    const farmerItem = result.element.cloneNode(true);
                    listItem.appendChild(farmerItem);
                    searchResultsList.appendChild(listItem);
                } else {
                    const listItem = document.createElement('li');
                    const submissionId = result.id || '';
                    const profileUrl = profileUrlBase + '?submission_id=' + submissionId 
                        + '&page_retour=' + currentPage
                        + '&view_mode=' + activeView
                        + '&search_query=' + encodeURIComponent(currentSearchQuery)
                        + '&search_page=' + searchCurrentPage;
                    
                    let categoriesHtml = '';
                    if (result.categories && result.categories.length > 0) {
                        categoriesHtml = '<p class="farmer-categories">' + result.categories.join(', ') + '</p>';
                    }
                    
                    let imageUrl = result.image || '';
                    if (imageUrl && !imageUrl.startsWith('http') && !imageUrl.startsWith('/')) {
                        imageUrl = root + imageUrl;
                    }
                    if (!imageUrl) {
                        imageUrl = root + 'media/mod_agriculteur_liste/images/default-farmer.jpg';
                    }
                    
                    const distance = result.distance || 0;
                    const formattedDistance = Number.isFinite(distance) ? distance.toFixed(1) : "-";
                    
                    listItem.innerHTML = `
                        <a href="${profileUrl}" class="farmer-item" 
                           data-id="${submissionId}"
                           data-name="${result.name || ''}"
                           data-owner="${result.owner || ''}"
                           data-coordinates="${result.coordinates || ''}">
                            <div class="farmer-content">
                                <img src="${imageUrl}" alt="${result.name || ''}" class="farmer-image" onerror="this.src='${root}media/mod_agriculteur_liste/images/default-farmer.jpg'">
                                <div class="farmer-info">
                                    <h3 class="farmer-name">${result.name || ''}</h3>
                                    <p class="farmer-owner">${result.owner || ''}</p>
                                    ${categoriesHtml}
                                </div>
                                <div class="farmer-distance" id="distance-search-${submissionId}">
                                    <p class="distance-value">${formattedDistance}</p>
                                    <p class="distance-unit">km</p>
                                </div>
                            </div>
                        </a>
                    `;
                    
                    searchResultsList.appendChild(listItem);
                }
            });
            
            updateSearchPagination();
        }
        
        listView.classList.remove('active-view');
        gridView.classList.remove('active-view');
        searchResultsContainer.style.display = 'block';
        if (listViewButton) listViewButton.classList.remove('active');
        if (gridViewButton) gridViewButton.classList.remove('active');
        
        document.querySelector('.top-navigation-bar').scrollIntoView({ behavior: 'smooth' });
        
        if (geolocateButton && navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    const userLat = position.coords.latitude;
                    const userLng = position.coords.longitude;
                    
                    searchResults.forEach(function(result) {
                        if (result.coordinates) {
                            const [farmLat, farmLng] = result.coordinates.split(',').map(parseFloat);
                            const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                            
                            const distanceElement = document.getElementById('distance-search-' + result.id);
                            if (distanceElement) {
                                const distanceValue = distanceElement.querySelector('.distance-value');
                                if (distanceValue) {
                                    distanceValue.textContent = distance.toFixed(1);
                                }
                            }
                        }
                    });
                },
                function(error) {
                    console.error('Erreur de géolocalisation:', error.message);
                }
            );
        }
    }
    
    function updateSearchPagination() {
        const paginationContainer = document.querySelector('.pagination-container');
        if (!paginationContainer) return;
        
        const totalSearchPages = Math.ceil(searchResults.length / searchItemsPerPage);
        
        if (searchResults.length <= 5 || totalSearchPages <= 1) {
            paginationContainer.style.display = 'none';
            return;
        }
        
        paginationContainer.style.display = 'flex';
        const pageInfo = paginationContainer.querySelector('.page-info');
        if (pageInfo) pageInfo.textContent = `Page ${searchCurrentPage} sur ${totalSearchPages}`;
        
        const pagination = paginationContainer.querySelector('.pagination');
        if (pagination) {
            pagination.innerHTML = '';
            
            if (searchCurrentPage > 1) {
                const firstPageButton = document.createElement('span');
                firstPageButton.className = 'page-item first';
                firstPageButton.innerHTML = '<i class="fas fa-angle-double-left"></i>';
                firstPageButton.style.cursor = 'pointer';
                firstPageButton.addEventListener('click', function() {
                    searchCurrentPage = 1;
                    displaySearchResults();
                });
                pagination.appendChild(firstPageButton);
                
                const prevPageButton = document.createElement('span');
                prevPageButton.className = 'page-item prev';
                prevPageButton.innerHTML = '<i class="fas fa-chevron-left"></i>';
                prevPageButton.style.cursor = 'pointer';
                prevPageButton.addEventListener('click', function() {
                    searchCurrentPage--;
                    displaySearchResults();
                });
                pagination.appendChild(prevPageButton);
            }
            
            const maxPagesToShow = 5;
            let startPage = Math.max(1, searchCurrentPage - Math.floor(maxPagesToShow / 2));
            let endPage = Math.min(totalSearchPages, startPage + maxPagesToShow - 1);
            startPage = Math.max(1, endPage - maxPagesToShow + 1);
            
            if (startPage > 1) {
                const firstPageLink = document.createElement('span');
                firstPageLink.className = 'page-item';
                firstPageLink.textContent = '1';
                firstPageLink.style.cursor = 'pointer';
                firstPageLink.addEventListener('click', function() {
                    searchCurrentPage = 1;
                    displaySearchResults();
                });
                pagination.appendChild(firstPageLink);
                
                if (startPage > 2) {
                    const ellipsis = document.createElement('span');
                    ellipsis.className = 'page-item ellipsis';
                    ellipsis.textContent = '...';
                    pagination.appendChild(ellipsis);
                }
            }
            
            for (let i = startPage; i <= endPage; i++) {
                const pageLink = document.createElement('span');
                pageLink.className = i === searchCurrentPage ? 'page-item active' : 'page-item';
                pageLink.textContent = i;
                
                if (i !== searchCurrentPage) {
                    pageLink.style.cursor = 'pointer';
                    pageLink.addEventListener('click', function() {
                        searchCurrentPage = i;
                        displaySearchResults();
                    });
                }
                
                pagination.appendChild(pageLink);
            }
            
            if (endPage < totalSearchPages) {
                if (endPage < totalSearchPages - 1) {
                    const ellipsis = document.createElement('span');
                    ellipsis.className = 'page-item ellipsis';
                    ellipsis.textContent = '...';
                    pagination.appendChild(ellipsis);
                }
                
                const lastPageLink = document.createElement('span');
                lastPageLink.className = 'page-item';
                lastPageLink.textContent = totalSearchPages;
                lastPageLink.style.cursor = 'pointer';
                lastPageLink.addEventListener('click', function() {
                    searchCurrentPage = totalSearchPages;
                    displaySearchResults();
                });
                pagination.appendChild(lastPageLink);
            }
            
            if (searchCurrentPage < totalSearchPages) {
                const nextPageButton = document.createElement('span');
                nextPageButton.className = 'page-item next';
                nextPageButton.innerHTML = '<i class="fas fa-chevron-right"></i>';
                nextPageButton.style.cursor = 'pointer';
                nextPageButton.addEventListener('click', function() {
                    searchCurrentPage++;
                    displaySearchResults();
                });
                pagination.appendChild(nextPageButton);
            }
        }
    }
    
    function hideSearchPagination() {
        const paginationContainer = document.querySelector('.pagination-container');
        if (paginationContainer) paginationContainer.style.display = 'none';
    }
    
    function checkSearchParamsFromReturnURL() {
        const urlParams = new URLSearchParams(window.location.search);
        const searchQuery = urlParams.get('search_query');
        const searchPage = urlParams.get('search_page');
        
        if (searchQuery) {
            currentSearchQuery = decodeURIComponent(searchQuery);
            searchCurrentPage = parseInt(searchPage) || 1;
            
            if (searchIndex) performSearch(currentSearchQuery);
            else loadSearchIndex();
            
            return true;
        }
        
        return false;
    }
    
    setTimeout(function() {
        if (!searchIndex) loadSearchIndex();
        checkSearchParamsFromReturnURL();
    }, 500);
    
    // ====================
    // = GÉOLOCALISATION
    // ====================
    
    if (geolocateButton) geolocateButton.addEventListener('click', getUserLocation);
    
    function getUserLocation() {
        if (!navigator.geolocation) {
            alert("La géolocalisation n'est pas supportée par votre navigateur.");
            return;
        }
        
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Localisation en cours...';
            geolocateButton.disabled = true;
        }
        
        navigator.geolocation.getCurrentPosition(
            function(position) {
                const userLat = position.coords.latitude;
                const userLng = position.coords.longitude;
                
                console.log('Position utilisateur obtenue:', userLat, userLng);
                
                if (distancesInfo) distancesInfo.style.display = 'block';
                
                if (isSearchMode) calculateSearchDistances(userLat, userLng);
                else calculateDistances(userLat, userLng);
                
                if (map) {
                    const userPosition = new google.maps.LatLng(userLat, userLng);
                    const userMarker = new google.maps.Marker({
                        position: userPosition,
                        map: map,
                        title: 'Votre position',
                        icon: { url: 'https://maps.google.com/mapfiles/ms/icons/blue-dot.png' }
                    });
                    
                    map.setCenter(userPosition);
                    map.setZoom(12);
                }
            },
            function(error) {
                console.error('Erreur de géolocalisation:', error.message);
                let errorMessage = "Impossible d'obtenir votre position.";
                
                switch(error.code) {
                    case error.PERMISSION_DENIED:
                        errorMessage += " Vous avez refusé l'accès à votre position.";
                        break;
                    case error.POSITION_UNAVAILABLE:
                        errorMessage += " Les informations de position ne sont pas disponibles.";
                        break;
                    case error.TIMEOUT:
                        errorMessage += " La demande de position a expiré.";
                        break;
                    case error.UNKNOWN_ERROR:
                        errorMessage += " Une erreur inconnue s'est produite.";
                        break;
                }
                
                alert(errorMessage);
                
                if (geolocateButton) {
                    geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
                    geolocateButton.disabled = false;
                }
            },
            { enableHighAccuracy: true, timeout: 10000, maximumAge: 0 }
        );
    }
    
    function calculateSearchDistances(userLat, userLng) {
        if (!searchResults || searchResults.length === 0) return;
        
        searchResults.forEach(function(result) {
            if (result.coordinates) {
                const [farmLat, farmLng] = result.coordinates.split(',').map(parseFloat);
                const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                result.distance = distance;
                
                const distanceElement = document.getElementById('distance-search-' + result.id);
                if (distanceElement) {
                    const distanceValue = distanceElement.querySelector('.distance-value');
                    if (distanceValue) distanceValue.textContent = distance.toFixed(1);
                }
            }
        });
        
        searchResults.sort((a, b) => {
            const distA = a.distance !== undefined ? a.distance : Infinity;
            const distB = b.distance !== undefined ? b.distance : Infinity;
            return distA - distB;
        });
        
        displaySearchResults();
        
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            geolocateButton.disabled = false;
        }
        
        showSuccessMessage();
    }
    
    function calculateDistances(userLat, userLng) {
        const farmerItems = document.querySelectorAll('.farmer-item');
        const gridItems = document.querySelectorAll('.grid-item');
        
        let distances = [];
        
        farmerItems.forEach(function(item) {
            const coordinates = item.getAttribute('data-coordinates');
            const id = item.getAttribute('data-id');
            
            if (coordinates) {
                const [farmLat, farmLng] = coordinates.split(',').map(parseFloat);
                const distance = calculateHaversineDistance(userLat, userLng, farmLat, farmLng);
                
                distances.push({ id: id, distance: distance, element: item });
                updateDistanceDisplay('distance-' + id, distance);
                updateDistanceDisplay('grid-distance-' + id, distance);
                item.setAttribute('data-distance', distance);
                
                const gridItem = document.querySelector(`.grid-item[data-id="${id}"]`);
                if (gridItem) gridItem.setAttribute('data-distance', distance);
            }
        });
        
        distances.sort((a, b) => a.distance - b.distance);
        
        const listContainer = document.getElementById('farmer-list-' + moduleId);
        if (listContainer) {
            const sortedItems = distances.map(item => item.element.parentNode);
            sortedItems.forEach(li => { if (li) listContainer.appendChild(li); });
        }
        
        const gridContainer = document.getElementById('farmer-grid-' + moduleId);
        if (gridContainer) {
            const gridItems = Array.from(document.querySelectorAll('.grid-item')).sort((a, b) => {
                return parseFloat(a.getAttribute('data-distance')) - parseFloat(b.getAttribute('data-distance'));
            });
            gridItems.forEach(item => gridContainer.appendChild(item));
        }
        
        if (geolocateButton) {
            geolocateButton.innerHTML = '<i class="fas fa-location-crosshairs"></i> Les plus proches';
            geolocateButton.disabled = false;
        }
        
        showSuccessMessage();
    }
    
    // ====================
    // = GOOGLE MAPS
    // ====================
    
    function loadGoogleMapsWithClusterer() {
        if (typeof google === 'undefined' || typeof google.maps === 'undefined') {
            const gMapScript = document.createElement('script');
            gMapScript.src = 'https://maps.googleapis.com/maps/api/js?key=AIzaSyAjsOoV0528RtrLFvylBrsypk0Z3qik5RI&callback=initGoogleMapAfterLoad';
            gMapScript.async = true;
            gMapScript.defer = true;
            
            const clusterScript = document.createElement('script');
            clusterScript.src = 'https://unpkg.com/@googlemaps/markerclustererplus/dist/index.min.js';
            
            window.initGoogleMapAfterLoad = function() {
                if (typeof MarkerClusterer === 'undefined') {
                    setTimeout(initGoogleMapAfterLoad, 100);
                } else {
                    initMap();
                }
            };
            
            document.head.appendChild(clusterScript);
            document.head.appendChild(gMapScript);
        } else {
            if (typeof MarkerClusterer === 'undefined') {
                const clusterScript = document.createElement('script');
                clusterScript.src = 'https://unpkg.com/@googlemaps/markerclustererplus/dist/index.min.js';
                clusterScript.onload = function() {
                    initMap();
                };
                document.head.appendChild(clusterScript);
            } else {
                initMap();
            }
        }
    }
    
    function initMap() {
        const reunionCenter = { lat: -21.115141, lng: 55.536384 };
        
        map = new google.maps.Map(document.getElementById('google-map'), {
            center: reunionCenter,
            zoom: 10,
            mapTypeId: google.maps.MapTypeId.ROADMAP,
            mapTypeControl: true,
            mapTypeControlOptions: {
                style: google.maps.MapTypeControlStyle.HORIZONTAL_BAR,
                position: google.maps.ControlPosition.TOP_RIGHT
            },
            zoomControl: true,
            zoomControlOptions: {
                position: google.maps.ControlPosition.RIGHT_CENTER
            },
            streetViewControl: true,
            streetViewControlOptions: {
                position: google.maps.ControlPosition.RIGHT_BOTTOM
            },
            fullscreenControl: true
        });
        
        markers = [];
        const processedIds = new Set(); // Ensemble pour suivre les IDs déjà traités
        
        // Sélectionner tous les éléments avec des coordonnées
        const farmerItems = document.querySelectorAll('[data-coordinates]');
        
        farmerItems.forEach(function(item) {
            const id = item.getAttribute('data-id');
            
            // Vérifier si on a déjà traité cet ID
            if (processedIds.has(id)) {
                return; // Ignorer cet élément
            }
            
            // Marquer l'ID comme traité
            processedIds.add(id);
            
            const coordinates = item.getAttribute('data-coordinates');
            const name = item.getAttribute('data-name');
            const owner = item.getAttribute('data-owner');
            const profileUrl = item.href;
            
            if (coordinates) {
                const [lat, lng] = coordinates.split(',').map(parseFloat);
                
                const marker = new google.maps.Marker({
                    position: { lat: lat, lng: lng },
                    map: map,
                    title: name,
                    icon: { url: 'https://maps.google.com/mapfiles/ms/icons/green-dot.png' }
                });
                
                const infowindowContent = document.createElement('div');
                infowindowContent.className = 'custom-infowindow';
                
                const contentDiv = document.createElement('div');
                contentDiv.className = 'infowindow-content';
                
                const titleElement = document.createElement('h3');
                titleElement.className = 'infowindow-title';
                titleElement.textContent = name;
                
                const subtitleElement = document.createElement('p');
                subtitleElement.className = 'infowindow-subtitle';
                subtitleElement.textContent = owner;
                
                const linkElement = document.createElement('a');
                linkElement.className = 'infowindow-link';
                linkElement.href = profileUrl;
                linkElement.textContent = 'Voir le profil';
                linkElement.style.color = 'white';
                linkElement.style.backgroundColor = '#3c8f44';
                linkElement.style.padding = '5px 10px';
                linkElement.style.borderRadius = '3px';
                linkElement.style.textDecoration = 'none';
                linkElement.style.display = 'inline-block';
                linkElement.style.marginTop = '5px';
                
                const imgContainer = document.createElement('div');
                imgContainer.className = 'infowindow-image-container';
                imgContainer.style.position = 'relative';
                imgContainer.style.width = '100%';
                imgContainer.style.paddingTop = '56.25%';
                imgContainer.style.overflow = 'hidden';
                
                const imgElement = document.createElement('img');
                imgElement.className = 'infowindow-image';
                imgElement.alt = name;
                imgElement.style.position = 'absolute';
                imgElement.style.top = '0';
                imgElement.style.left = '0';
                imgElement.style.width = '100%';
                imgElement.style.height = '100%';
                imgElement.style.objectFit = 'cover';
                
                const existingImg = item.querySelector('img');
                if (existingImg && existingImg.src) imgElement.src = existingImg.src;
                
                imgContainer.appendChild(imgElement);
                contentDiv.appendChild(titleElement);
                contentDiv.appendChild(subtitleElement);
                contentDiv.appendChild(linkElement);
                infowindowContent.appendChild(imgContainer);
                infowindowContent.appendChild(contentDiv);
                
                const infowindow = new google.maps.InfoWindow({
                    content: infowindowContent,
                    maxWidth: 300
                });
                
                marker.addListener('click', function() {
                    markers.forEach(function(m) {
                        if (m.infowindow) m.infowindow.close();
                    });
                    infowindow.open(map, marker);
                });
                
                marker.infowindow = infowindow;
                markers.push(marker);
            }
        });
        
        if (markers.length > 0) {
            markerCluster = new MarkerClusterer(map, markers, {
                imagePath: 'https://developers.google.com/maps/documentation/javascript/examples/markerclusterer/m',
                gridSize: 50,
                maxZoom: 15
            });
        }
        
        // Log pour debugging
        console.log('Nombre total de marqueurs créés:', markers.length);
    }
    
    if (mapViewButton) {
        mapViewButton.addEventListener('click', function() {
            if (!map) loadGoogleMapsWithClusterer();
        });
    }
    
    updatePaginationLinks(activeView);
    checkSearchParamsFromReturnURL();
});